/* ****************************************************************************
*
*	Module for managing the SRAM 23LC512-I/SN connected through SPI to the 
*	ATmega328P.
*
*	(c) Juan Antonio Fernandez Madrigal, 2015-2018
*
*
*	This module does not use any interrupts nor require any other module.
*
*	The block operations can be compiled in C version or in Assembler version
*	(optimized, by default). To switch to C version, pass the symbol definition
*	"-D SRAMBLOCKINC" to the compiler.
*
*	Code can be generated either for having the SRAM device SS in B2 or in D3 
*	(in the latter case, B2 is also internally used, anyway, but not for SS). 
*	By default, code is generated for SS in D3. If your SRAM device connects to
*	B2, pass the symbol definition "-D SS_B2" to the compiler.
*
*	The digital pins used by this module are:
*
*		MOSI -> PB3
*		MISO -> PB4
*		SCK -> PB5 
*		SS -> either PB2 or PD3, but PB2 is always used internally,
*			  independently on the actual pin used for SS.
*
*	NOTE: SPI works as follows: in an event A both master and slave put new data
*	in the corresponding output lines (setup); in an event B both sample the
*	data of the	corresponding input lines. Events A and B are edges of the clock
*	signal. For example, if CPOL=1, CPHA=1, then A=falling edge and B=rising
*  edge, which is the case of the SRAM chip.
*
*******************************************************************************/

#ifndef SRAM23LC512
#define SRAM23LC512

/* ------------------------ Basic routines -------------------------- */

unsigned SRAMinitiate(void);
/* Configure initially the SRAM for being used with SPI and return the number of
words (2-bytes) available in the SRAM chip. 
There is no need to de-configure at the end.*/

void SRAMwriteblock(unsigned address, const unsigned char *block,
					unsigned length);
/* Write into ADDRESS and next, LENGTH bytes taken from BLOCK.
Including the call to this routine and without interrupts enabled, the
transfer speed is around 5 microsecs per byte (200KHz) */

void SRAMreadblock(unsigned address, unsigned char *block, unsigned length);
/* Read from ADDRESS up, LENGTH bytes and store them into BLOCK.
Including the call to this routine and without interrupts enabled, the
transfer speed is around 5.3 microsecs per byte (190KHz) */

/* ------------------------ Additional routines -------------------------- */

void SRAMsendinstr(const unsigned char *instr, unsigned char lengthinstr,
				   unsigned char *resp, unsigned char lengthresp);
/* Send an individual instruction of LENGTHINSTR bytes to the SRAM chip, taken 
from INSTR, and after that waits for LENGTHRESP bytes to be received, storing
them into RESP. The instruction must be one recognized in the SRAM chip 
communication protocol. */

#endif
