/* ****************************************************************************
*
*	Simple RS232 library for ATmega328P
*
*	(c) Juan-Antonio Fernandez-Madrigal, 2015-2018
*
*******************************************************************************/

/* 

  There are a number of bauds for which their exact frequencies can be achieved
by the ATmega with a 16MHz clock. According to table 19-1 (p. 179) of the 
datasheet, there are 2 ways to configure the prescaler of the USART in 
asynchronous transmissions: either the U2X0 bit is 0 or 1. In the former case, 
the prescaler is obtained by the formula presc = 1000000/bauds - 1, while in the 
latter case it is presc = 2000000/bauds - 1. With the following Matlab code we 
can calculate the prescalers and bauds that can be exactly computed (without 
decimals) through these formulae, and it turns out that they are exactly the 
same except for the case of 2000000 bauds, which can only be obtained with 
U2X0 = 1:

	bauds = [1:2000000];
	ubbrns = 16000000./(bauds*8)-1;
	inds = find(ubbrns == floor(ubbrns));
	ubbrnsbis = 16000000./(bauds*16)-1;
	indsbis = find(ubbrnsbis == floor(ubbrnsbis));
	baudsexact = union(bauds(inds),bauds(indsbis));

  The resulting bauds are: 

	1 2 4 5 8 10 16 20 25 32 40 50 64 80 100 125 160 200 
	250 320 400 500 625 800 1000 1250 1600 2000 2500 3125 
	4000 5000 6250 8000 10000 12500 15625 20000 25000 31250 
	40000 50000 62500 100000 125000 200000 250000 500000 1000000 
	2000000 (<- with UX20 = 1)

  The most common bauds in use are: 

	baudsusual = [2400 4800 9600 14400 19200 28800 38400 57600 76800 115200 ...
				  230400 250000 500000 1000000 2000000];

  The closest bauds that can be exactly achieved are:

	baudsclosest = zeros(1,length(baudsusual));
	for (f=1:length(baudsusual)); [m,i] = min(abs(baudsexact-baudsusual(f)));... 
		baudsclosest(f) = baudsexact(i(1)-1); end

	baudsclosest = [2000 4000 8000 12500 16000 25000 31250 50000 62500 ...
					100000 200000 200000 400000 500000 1000000];
*/

#include "rs232atmega.h"

#include <stdlib.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <string.h>

#ifndef F_CPU
	#error RS232 module cannot compile without the F_CPU symbol defined
#endif

#if F_CPU != 16000000L
	#warning RS232 module only compile correctly if F_CPU = 16000000L
#endif

/* ------------------ RS232 constants, types, variables and macros ---------- */

// RS232 receiving status
typedef enum {
				RS232_RS_WAITING_DATA=0, // still receiving data
				RS232_RS_OVERFLOW, // too long command received (excess dropped)
				RS232_RS_END, // end of command received
				RS232_RS_POSTEND, // end received and then more data received
				RS232_RS_PAUSE	// external program has paused ISR work
			 } RS232RecStatus;

// RS232 input buffer (NULL if not used)
static volatile char *rs232buffer;
// RS232 input buffer size
static volatile unsigned char rs232buffersize;
// RS232 current position in the input buffer
static volatile unsigned char rs232bufferpos;
// RS232 input buffer state
static volatile unsigned char rs232bufferstate=RS232_RS_PAUSE;
// RS232 input buffer end char
static volatile char rs232endchar;
// RS232 input data
static volatile RS232InputReport rs232inputreport;



/* ------------------ RS232 ISRs --------------- */

ISR(USART_RX_vect)
// USART rx complete; a byte is waiting to be read from the usart
// This interrupt has higher priority than ADC, blocking the ADC until finished
// It has higher priority than the other usart isrs.
{
	// read hardware
	rs232inputreport.lasterr=UCSR0A & ((1 << FE0)|(1 << DOR0)|(1 << UPE0));
	rs232inputreport.lastinputchar=UDR0;

	// enable other interrupts while we do the less strict part of the process;
	// do not enable this very interrupt for avoiding recursion
	UCSR0B &= (~(1 << RXCIE0));
	sei();

	// process data and status
	switch ((RS232RecStatus)rs232bufferstate)
	{
		case RS232_RS_WAITING_DATA: // a new char is available while waiting
		{
			rs232buffer[rs232bufferpos++]=rs232inputreport.lastinputchar;
			if (rs232inputreport.lastinputchar==rs232endchar)
				rs232bufferstate=(unsigned char)RS232_RS_END;
			else if (rs232bufferpos==rs232buffersize)
				rs232bufferstate=(unsigned char)RS232_RS_OVERFLOW;
			break;
		}
		case RS232_RS_OVERFLOW: // receiving too long command
		{
			rs232inputreport.bufov++;
			if (rs232inputreport.lastinputchar==rs232endchar)
				rs232bufferstate=(unsigned char)RS232_RS_END;
			break;
		}
		case RS232_RS_END: // received new data after the end of previous
		{
			rs232bufferstate=RS232_RS_POSTEND; // data lost
			break;
		}
		default: break; // all other cases do nothing and lose data
	}

	// restore interrupts
	cli();
	UCSR0B |= (1 << RXCIE0);
}

/* ------------------ RS232 routines --------------- */

static void RS232_PrepareToRead(void)
// Prepare the input rs232 system to read the next command, initializing the
// status and losing any data already read.
{
	rs232bufferstate=RS232_RS_PAUSE; // disable ISR work
	if (rs232buffer!=NULL)
	{
		// prepare variables
		rs232bufferpos=0;
		rs232inputreport.bufov=0;
		rs232inputreport.lasterr=0;
		rs232inputreport.lastinputchar=0;
		// start receiving
		rs232bufferstate=RS232_RS_WAITING_DATA; // must be at the end
	}
}

unsigned long RS232_Init(char *buf, unsigned char bufsize, char endch, 
			 			 unsigned long bauds)
{
	// backup interrupt status and disable interrupts
	unsigned char prevints=(SREG & (1 << 7));
	if (prevints) cli();

	unsigned long actualbauds;
	unsigned presc;
	unsigned char x2 = 0;
	if (bauds>2000000UL) bauds = 2000000UL; // cannot achieve more with 16MHz
	if (bauds > 1000000UL) // maximum bauds with U2X0 == 0
	{
		presc = (unsigned long)F_CPU / (8UL * bauds) - 1;
		actualbauds = (unsigned long)F_CPU/(8UL * ( (unsigned long)presc + 1 ));
		x2 = 1;
	}
	else // U2X0 = 0
	{
		presc = (unsigned long)F_CPU / (16UL * bauds) - 1;
		actualbauds = (unsigned long)F_CPU/(16UL *( (unsigned long)presc + 1 ));
	}
	if (x2) UCSR0A |= (1 << U2X0);
	else UCSR0A &= ~(1 << U2X0); 
	UCSR0A &= ~(1 << MPCM0);
	UBRR0H = (presc >> 8);
	UBRR0L = (presc & 0xff);

/*
	// UBBRn = 25 for 38400 bauds (p. 203, datasheet)
	// UBBRn = 8 for 115200 bauds
	// UBBRn = 1 and U2X0 = 1 for 1000000 bauds
	// UBBRn = 0 and U2X0 = 1 for 2000000 bauds
 	UBRR0H = (25 >> 8); 
	UBRR0L = (25 & 0xff); // this updates the baudrate generator (p. 199)

	// no x2 speed, no multicomms (p. 196)
	UCSR0A &= (~(1 << U2X0)) & (~(1 << MPCM0));
*/
	// enable isr vects, enable tx and rx hardware, 8 bits
	UCSR0B = (1 << RXCIE0) | (1 << RXEN0) | (1 << TXEN0);

	// 8 bits, asynchronous, no parity, 1 stop
	UCSR0C = (3 << UCSZ00);

	// flush hardware buffered data
//	unsigned char aux;
	while (UCSR0A & (1 << RXC0))
	{ // in assembler for avoiding the "unused after set" warning on AUX
		__asm__ __volatile__ (
			"push r17"		" \n\t"
			"push r30"		" \n\t"
			"push r31"		" \n\t"
			"ldi r30,0xc6"	" ; Z pointing to UDR0 register \n\t"
			"clr r31"		" \n\t"
			"ld r17,Z"		" ; read UDR0 and discard it \n\t"
			"pop r31"		" \n\t"
			"pop r30"		" \n\t"
			"pop r17"		" \n\t"
		);
//		aux=UDR0;
	}

	// prepare software reading buffer
	rs232buffer=buf;
	rs232buffersize=bufsize;
	rs232endchar=endch;
	RS232_PrepareToRead();

	// resume interrupt status
	if (prevints) sei();

	return(actualbauds);
}

void RS232_Finish(void)
// Disable the hw/sw system for rs232 communications.
// Interrupts should be disabled previously.
{
	// disable further isr work
	rs232bufferstate=RS232_RS_PAUSE;
	// disable isr, tx and rx hardware
	UCSR0B = 0;
}

void RS232_Read(RS232InputReport *inpcp)
// Wait until the current command is completed, either with or
// without errors, then copy the resulting report into INPCP, clearing the
// status of the input receiver in order to receive further commands.
// (NOTE: Infinite waiting is possible, if no command is received)
{
	if (rs232buffer!=NULL)
	{
		// wait pending receptions
		while ((RS232RecStatus)rs232bufferstate<RS232_RS_END);
		// disable ISR work
		rs232bufferstate=RS232_RS_PAUSE;
		if (inpcp!=NULL) memcpy(inpcp,(const void *)&rs232inputreport,sizeof(RS232InputReport));
		// enable again ISR work
		RS232_PrepareToRead();
	}
}

void RS232_Send(const char *c, unsigned char endchar)
// Send command in C and return after the whole transmission
{
	if (c==NULL) return;
	// send command
	unsigned f=0;
	while (c[f]!=endchar)
	{
		while ( !( UCSR0A & (1<<UDRE0)) );
		UDR0=c[f++];
	}
	while ( !( UCSR0A & (1<<UDRE0)) );
	UDR0=endchar;
}

void RS232_SendNoEnd(const char *c, unsigned char endchar)
{
	if (c==NULL) return;
	// send command
	unsigned f=0;
	while (c[f]!=endchar)
	{
		while ( !( UCSR0A & (1<<UDRE0)) );
		UDR0=c[f++];
	}
	while ( !( UCSR0A & (1<<UDRE0)) );
}
