function l = simulationlightsensor(bitmapgray,xul,yul,wpix,hpix,...
                                   xl,yl,rl,sl)
% Simulate a reading from reflected light of a light sensor placed at XL,YL 
% with a cylinder radius of RL and noise (sigma) in the support [0,1] of
% SL. The pose of the sensor is assumed in the universal frame of the
% floor, that follows the right hand rule.
% The floor is assumed to have a gray pattern given in BITMAPGRAY, a matrix
% with a picture of the light levels (from 0 to 255) with the left-hand
% rule used for its dimensions (width is X and height is Y). That bitmap is
% placed on the floor with the center of its top-left corner pixel at 
% XUL,YUL, and with the width and height of its pixels as in WPIX,HPIX.
% If the sensor is outside the bitmap, a value of NaN is returned.
% Otherwise, a value in [0,1] is returned: 0-> dark, 1-> light.

    l = NaN;
    if isempty(bitmapgray)
        return;
    end
    
    [h,w] = size(bitmapgray); % rows are height, columns are width
    
    xbr = xul + wpix * w;
    ybr = yul - hpix * h;
    if ~pointInHull([xl,yl],[xul ybr xbr yul])
        return;
    end
    
    % hull of the square area that contains the circle of the sensor on the
    % plane
    minxl = xl - rl;
    maxxl = xl + rl;
    minyl = yl - rl;
    maxyl = yl + rl;
    
    % the same hull in indexes of the bitmap
    [~,indcol_minxl,indrow_minyl] = pixelFromPoint(minxl,minyl,xul,yul,wpix,hpix,h,w,1);
    [~,indcol_maxxl,indrow_maxyl] = pixelFromPoint(maxxl,maxyl,xul,yul,wpix,hpix,h,w,1);
    hullpix = [min(indrow_minyl,indrow_maxyl) min(indcol_minxl,indcol_maxxl) ...
               max(indrow_minyl,indrow_maxyl) max(indcol_minxl,indcol_maxxl)];
           
    % simulate the sensor in that hull
    lm = sum(sum(double(bitmapgray(hullpix(1):hullpix(3),hullpix(2):hullpix(4))))) / ...
         ((hullpix(3)-hullpix(1)+1)*(hullpix(4)-hullpix(2)+1));
    l = lm/255;
    if sl > 0
        l = l + normrnd(0,sl);
        if l < 0
            l = 0;
        elseif l > 1
            l = 1;
        end
    end

end

function in = pointInHull(p,hull) 
% hull is minx,miny,maxx,maxy

    if (p(1) > hull(1)) && (p(1) < hull(3)) && ...
       (p(2) > hull(2)) && (p(2) < hull(4))
        in = 1;
    else
        in = 0;
    end
    
end

function [inside,indcol,indrow] = pixelFromPoint(x,y,xul,yul,wpix,hpix,h,w,adjust)
% Return indices of the pixel in the bitmap defined by XUL,YUL,WPIX,HPIX,H,W 
% given a point in the plane X,Y
% Return INSIDE == 1 if the point is inside the bitmap
% If ADJUST == 1, when the point is outside the bitmap, adjust its indices
% to the closest edge of the bitmap.

    indcol = floor((x-xul)/wpix) + 1;
    indrow = floor((yul-y)/hpix) + 1;
    if (indcol >= 1) && (indcol <= w) && (indrow >= 1) && (indrow <= h)
        inside = 1;
    else
        inside = 0;
        if adjust
            if indcol < 1
                indcol = 1;
            end
            if indcol > w
                indcol = w;
            end
            if indrow < 1
                indrow = 1;
            end
            if indrow > h
                indrow = h;
            end
        end
    end 

end