% Lego class for simulation and control of Lego Mindstorms Robots.
% -----------------------------------------
% (c) Juan-Antonio Fernández-Madrigal
% System Engineering and Automation Dpt.
% University of Malaga (Spain)
% http://babel.isa.uma.es/jafma
% 
% This simulation has some realistic features:
%
%   - Motion of the wheels has certain dynamics: rotational velocities
%   follow a first order time response.
%
%   - Sonar has longitudinal noise in the sonar measurements. For advanced
%   users (see defineRobot() method), even an angular noise and a beam-like
%   thickness may be simulated for advanced users.
%
%   - Reflected light sensor has noise and a certain radius from which it
%   takes the average reflected light.
%
%   - For advanced users (see defineRobot()), the wheels may have different
%   radii.
%
% The functions of this library allow the Matlab programmer to command
% basic actions in a simulated Lego Mindstorm robot and reading its 
% sensors. NOTE: Only the methods in the "Actuators and sensors" section
% below are available in a real robot; the rest of them provide the exact
% state of the system, only available in the simulated robot.
%
% The methods are (please use "help Lego.<method>" to get more details):
%
%   --------- Creation / deletion of the object ------------
%
%	Lego -> create the Lego simulation object.
%
%   --------- Definition of the simulation environment ------------
%
%   defineRobot -> define the physical features of the robot
%
%   getRobotDef -> return the current robot definition
%
%   defineWalls -> define the walls (obstacles) in the environment
%
%   getWalls -> return the current definition of walls
%
%   loadLightMap -> load a new light map on the floor from an image file.
%
%   --------- General state of the robot ------------
%
%   changePose -> change the ground-truth pose of the robot at once
%
%   getPose -> get the current ground-truth pose of the robot
%
%   getAngWheel -> get the current angle of a wheel, without the
%                  discretization of the encoders.
%
%   collision -> calculate if the robot is colliding with the walls.
%
%   --------- Actuators and sensors ------------
%
%	setMotor -> change the power of one motor
%
%   getMotor -> read the current power in one motor
%
%	readEncoder -> read one encoder
%
%	readUltrasonic -> read the sonar
%
%	readLight -> read the light sensor
%
%   readGyro -> read the gyroscope sensor
%
%   resetGyro -> reset the measurements of the gyroscope sensor
%
%   measureRobot -> simulate the manual measurement of different parameters
%                   of the robot with a rule.
%
%   --------- Simulation ------------
%
%   getSimTime -> return the current simulation time
%
%   simulate -> simulate the robot for a given duration
%
%   getFigure -> get the handler of the figure where the robot is being
%                drawn.
%
%   redraw -> redraw the simulator figure.
%
%   changeDrawEnv -> change the area of the environment to draw to a fixed
%                    one.
%
%   changeDrawEnvMov -> change the area of the environment to draw to one
%                    that moves with the robot.
%
%   setUltrasonicDrawing -> enable or disable the drawing of the ultrasonic
%                           sensor data.
%
%   setLightDrawing -> enable or disable the drawing of the light sensor
%                      device and of the light map (if any).

% VERSION HISTORY
%
% v1.2.5 - November 2023
%
%   -Fixed a bug in the wall detector.
%
% v1.2.4 - March 2022
%
%	-Fixed the aspect ratio of simulator figure.
%	-Fixed the documentation of the constructor.
%
% v1.2.3 - March 2021
%
%	-Improved the documentation of each method.
%
% v1.2.2 - July 2020
%
%	-Fixed the use of the degree symbol in the title of the window (previously,
%	it was not a standard character).
%
% v1.2.1 - June 2020:
%
%	-Fixed a bug in the wall collision method.
%   -Slightly changed the help of the measureRobot() method and its
%   calculations for providing a more accurate offsphi.
%
% v1.2.0 - April 2020:
%
%   -Added a simulated EV3 gyro sensor.
%
% v1.1.0 - April 2020:
%
%   -Added a new method to set the viewport to one that moves with the
%   robot.
%   -More robust use of the object if the figure is already closed.
%
% v1.0.1 - April 2020:
%
%   -Compatibility with Matlab 2016b+
%   -Changed robot poses to column vectors
%   -Changed light readings to integers in range [0..100]
%   -Changed sonar readings to integers in range [0..MAXSONAR] cm
%   -Changed encoder readings to integer values
%   -Changed motor power inputs to force integer values
%
% v1.0.0 - March 2020

classdef Lego < handle
    % It is a handle for being modified when passed to a function; copies
    % will refer to the same object, but you can create any number of
    % objects.
    
    properties (Constant, Access = public)
    
        VER = '1.2.5';
        
        SONARPLACE_FRONT = 0;
        SONARPLACE_CROSS = 1;
        
        MAXSONAR = 255;
        MAXLIGHT = 100;
        
    end
    
    properties (Access = private)
    
        mode;

        % simulation properties
        
        feats;
        environ; % struct with these fields:
                 %  .mode -> 0 for fixed, 1 for moving
                 %      .minx, .miny, .maxx, .maxy -> parameters of fixed
                 %      .radiusx, .radiusy -> parameters of moving
        gt_pose;
        robshape;
        walls;
        light;
        disp;
        drawsonar;
        drawlight;
        
        t;
        angs; % 1->l, 2->r, in radians
        alphas; % idem, in rad/s
        motors; % idem, in -100..100, integer
        
        % These sensor values have already their noise embedded:
        
        encs; % idem, in degrees, not discretized
        lastsonar; % in meters, NaN if none, not discretized.
        lastlight; % in [0,1], NaN if none, not discretized.
        lastgyro; % 3-cell vector: (1) -> angle (rad) , (2) -> w (rad/s),
                  % (both are NaN if none, not discretized), (3) -> ground-
                  % truth orientation of the robot used as the reference
                  % to calculate increments of gyro angles.
               
    end
    
    
    methods (Access = public)
        
        % ----- Constructor

        function obj = Lego(sonarplace)
        % Construct a new simulator. 
        %
		% 	s = Lego(sonarplace);
        %
        % SONARPLACE must be one of the possible places for the sonar
        % sensor, i.e.,  Lego.SONARPLACE_FRONT or Lego.SONARPLACE_CROSS.
        
            if (nargin==0)
                sonarplace = Lego.SONARPLACE_CROSS;
            end
            switch (sonarplace)
                case 0
                    rTs = [1  0 0 -0.1; ...
                           0  1 0 0; ...
                           0  0 1 0 ;    ...
                           0  0 0 1];
                case 1
                    rTs = [0 -1 0 -0.05; ...
                           1  0 0 0.015; ...
                           0  0 1 0 ;    ...
                           0  0 0 1];
                otherwise
                    error('Invalid sonar place');
            end            
        
            % default physical features (see the defineRobot() method)
            obj.defineRobot(struct('d',0.114,... % wheel distance
                               'r',[0.028,0.028],... 
                               'hull',[0.17,0.12,0.05],... 
                               'rTs',rTs, ... 
                               'semiangle_sonar',deg2rad(0),...
                               'sigmas_sonar',[0.005,0],...
                               'rTl',[1 0 0 0.05; ...
                                      0 1 0 0; ...
                                      0 0 1 0; ...
                                      0 0 0 1],...
                               'radius_light',0.005,...
                               'sigma_light',0.05));
                           
            % initial configuration
            obj.t = 0;
            obj.gt_pose = [0,0,0].';
            obj.encs = [0,0];      
            obj.angs = [0,0];
            obj.alphas = [0,0];
            obj.motors = [0,0];
            obj.lastsonar = NaN;
            obj.lastlight = NaN;
            obj.resetGyro();
            
            obj.walls = [];
            obj.light = struct('lightmap',[],... % h x w x 3 uint8 bitmap with 0..255 in each component
                               'xul',0,...  % top-left x-coordinate on the floor
                               'yul',0,...  % top-left y-coordinate on the floor
                               'wpix',0,... % width of pixels on the floor
                               'hpix',0);   % height of pixels on the floor
                                             
            obj.environ = struct('mode',1,...
                                 'radiusx',0.5,...
                                 'radiusy',0.5);
            obj.disp = figure();
            obj.drawsonar = 2;
            obj.drawlight = 1;
            obj.drawAll();
           
        end
        
        % ----- Destructor
        
        function delete(obj)

            if obj.checkFigure()
                close(obj.disp);
            else
                fprintf('No simulation figure to delete.\n');
            end
            fprintf('Deleted Lego object.\n');
            
        end
        
        % ----- Public methods

        function defineRobot(obj,features)
        % Define the physical features of the robot. By default, the robot
        % has some built-in values in these features.
        % 
		% 	s.defineRobot(features);
		%
        % FEATURES must be a struct with these fields:
        %   .d -> distance between both wheels, in meters
        %   .r -> vector with the 2 radii of the wheels, in meters
        %   .hull -> vector with 3 values: longitudinal size of the robot
        %            body; transversal size of the robot body; distance
        %            from the center of movement to the front of the body.
        %            All in meters.
        %   .rTs ->  homogeneous transform from sonar system to robot
        %            system. The robot system has the x axis pointing
        %            towards the orientation of the robot and the z axis
        %            upwards (origin at the middle of the line connecting
        %            both wheels). The sonar system has the x axis pointing
        %            in the orientation of the sonar beam and the z axis
        %            upwards. The z displacement between both systems is
        %            ignored in this simulator. Right-hand rule is used for
        %            getting y axes. All data must be in meters.
        %            NOTE: Being this a 3D matrix, the z axis information 
        %            and any rotation out of the plane are ignored:
        %            the sonar beam is assumed to be contained
        %            in the plane of the environment.
        %   .semiangle_sonar -> half of the angle of the beam of the sonar,
        %            where any obstacle will be detected without problems
        %            (in rads).
        %   .sigmas_sonar -> sigmas for the gaussian noises in the 
        %            longitudinal (1) and angular (2) parameters of the 
        %            sonar beam, in meters.
        %   .rTl -> homogeneous transform from light sensor system to 
        %           robot system. The light sensor system has its x axis
        %           pointing towards the outside of the robot and its z
        %           axis upwards. The rest is like in the .rTs field.
        %           NOTE: The light sensor is assumed to point to the
        %           floor; therefore, the z axis information and any
        %           rotation out of the floor plane are ignored.
        %   .radius_light -> radius in meters of the light emitting
        %           cylinder.
        %   .sigma_light -> sigma for the noise in the light sensor, for
        %           light measurements in the support [0,MAXLIGHT].
        
            if ~isfield(features,'d')
                error('Wheel distance must be in field "d"');
            end
            Lego.checkIsPositive(features.d,'.d field');
      
            if ~isfield(features,'r')
                error('Wheel radii must be in field "r"');
            end
            Lego.checkIsVectorOfN(features.r,2,'.r field');
            Lego.checkIsPositive(features.r(1),'.r(1)');
            Lego.checkIsPositive(features.r(2),'.r(2)');

            if ~isfield(features,'hull')
                error('Robot convex hull must be in field "hull"');
            end
            Lego.checkIsVectorOfN(features.hull,3,'.hull field');
            Lego.checkIsPositive(features.hull(1),'.hull(1)');
            Lego.checkIsPositive(features.hull(2),'.hull(2)');
            Lego.checkIsPositive(features.hull(3),'.hull(3)');
            
            if ~isfield(features,'rTs')
                error('Sonar to robot transform must be in field "rTs"');
            end
            Lego.checkIsHT(features.rTs,'.rTs field');
            
            if ~isfield(features,'semiangle_sonar')
                error('The sonar beam half-angle must be in field "semiangle_sonar"');
            end
            Lego.checkIsPositive(features.d,'.semiangle_sonar field');
            
            if ~isfield(features,'sigmas_sonar')
                error('Sonar noises must be in field "sigmas_sonar"');
            end
            Lego.checkIsVectorOfN(features.sigmas_sonar,2,'.sigmas_sonar field');

            if ~isfield(features,'rTl')
                error('Light to robot transform must be in field "rTl"');
            end
            Lego.checkIsHT(features.rTl,'.rTl field');

            if ~isfield(features,'radius_light')
                error('Radius of the light cylinder must be in field "radius_light"');
            end
            Lego.checkIsPositive(features.radius_light,'.radius_light');

            if ~isfield(features,'sigma_light')
                error('Noise of the light sensor must be in field "sigma_light"');
            end
            Lego.checkIsNumScalar(features.sigma_light,'.sigma_light');

            obj.feats = features;
            obj.updateRobShape();
            
            fprintf('Robot physical features changed ok.\n');
        
        end
        
        function features = getRobotDef(obj)
        % Get the current definition of robot physical features
        %
		% 	features = s.getRobotDef();
		%
        % FEATURES will be as in the "defineRobot" method.
        
            features = obj.feats;
        
        end
        
        function defineWalls(obj,ws)
        % Change the current wall definition by the one in WS.
        %
        % WS is a matrix with as many rows as walls and 4 columns with the
        % (x,y) coordinates of the ends of each wall.
        
            if (~isempty(ws))
                [~,nc] = size(ws);
                if (nc ~= 4)
                    error('WS must have 4 columns');
                end
            end
            obj.walls = ws;
            
            obj.drawAll();
            
            fprintf('Walls definition changed ok.\n');
            
        end

        function ws = getWalls(obj)
        % Get the current definition of walls
        % 
		% 	ws = s.getWalls();
        %
        % WS will be as in the "defineWalls" method.
        
            ws = obj.walls;
        
        end
        
        function loadLightMap(obj,filename,xul,yul,wpix,hpix)
        % Load an image that defines the light map on the floor.
        % 
		% 	s.loadLightMap(filename,xul,yul,wpix,hpix);
        %
        % FILENAME is the file of the image, that can be any of the types
        % that Matlab is able to load.
        % XUL,YUL are the desired coordinates on the universal system of
        % the upper-left corner of the picture, in meters.
        % WPIX and HPIX are the dimensions in meters of each pixel (width
        % and height).
        
            Lego.checkIsNumScalar(xul,'xul');
            Lego.checkIsNumScalar(yul,'yul');
            Lego.checkIsPositive(wpix,'wpix');
            Lego.checkIsPositive(hpix,'hpix');
        
            [im,cm] = imread(filename);
            ims = size(im);
            if (length(ims) < 2)
                error('Less than 2 dimensions in image file');
            end
            height = ims(1);
            width = ims(2);
            switch (length(ims))
                case 3 % rgb true color, with components from 0 to 255
                    
                    if ims(3) ~= 3
                        error('Cannot handle %d color planes in image file',ims(3));
                    end
                    rgbbitmap = im2double(im); % rgb with comps from 0 to 1
                    graybitmap = uint8((squeeze(rgbbitmap(:,:,1)) + ...
                                        squeeze(rgbbitmap(:,:,2)) + ...
                                        squeeze(rgbbitmap(:,:,3))) / 3 * 255);
                    obj.light.lightmap = uint8(zeros(height,width,3));
                    obj.light.lightmap(:,:,1) = graybitmap;
                    obj.light.lightmap(:,:,2) = graybitmap;
                    obj.light.lightmap(:,:,3) = graybitmap;
                    
                case 2 % either grayscale or indexed image
                    
                    if isempty(cm) % grayscale
                        
                        obj.light.lightmap = uint8(zeros(height,width,3));
                        obj.light.lightmap(:,:,1) = ims;
                        obj.light.lightmap(:,:,2) = ims;
                        obj.light.lightmap(:,:,3) = ims;                        
                        
                    else % indexed
                        
                        Lego.unimplemented();
                        
                    end
                    
                otherwise
                    error('Cannot handle that number of dimensions (%d) in image file',length(ims));
            end
            obj.light.xul = xul;
            obj.light.yul = yul;
            obj.light.wpix = wpix;
            obj.light.hpix = hpix;
            
            obj.drawAll();
            
            fprintf('Light map loaded ok.\n');
            
        end
        
        function changePose(obj,p)
        % Change the ground-truth pose of the robot to P. It also reset 
        % the gyro.
        % 
		% 	s.changePose(p);
        %
        % P must be a column vector with 3 real numbers: x,y,theta, being
        % (x,y) in meters and theta in radians.
        
            Lego.checkIsVectorOfN(p,3,'The pose');
            [r,~] = size(p);
            if (r ~= 3)
                error('Poses must be column vectors.');
            end
            obj.gt_pose = p;
            
            obj.resetGyro();
            
            obj.drawAll();
            
            fprintf('Real robot pose changed ok.\n');
        
        end
        
        function p = getPose(obj)
        % Get the current ground-truth pose of the robot.
        % 
		% 	p = s.getPose();
        %
        % P is a column vector as in the "changePose" method.
        
            p = obj.gt_pose;
        
        end
        
        function a = getAngWheel(obj,wheel)
        % Get the exact angle in radians of the position of wheel WHEEL,
        % without the discretization of the encoder.
        % 
		% 	a = s.getAngWheel(wheel);
        %
        % WHEEL must be 'L' or 'R'.
        
            which = Lego.checkMotor(wheel);
            a = obj.angs(which);
        
        end
        
        function [c,i] = collision(obj)
        % Calculate if the robot is colliding with the walls using the
        % convex hull of the robot.
        % 
		% 	[c,i] = s.collision();
        %
        % C will be 1 for a collision or 0 for no collision. In the former
        % case, I will be the index of the first wall that produces such a
        % collision.
        
            c = 0;
            i = NaN;
            if (isempty(obj.walls))
                return;
            end
            
            uTr = obj.robot2univMatrix();
            rTu = [ uTr(1:3,1:3).' -(uTr(1:3,1:3).')*uTr(1:3,4) ; ...
                    0 0 0 1];
            
            [nw,~] = size(obj.walls);
            hulllims = [ -(obj.feats.hull(1)-obj.feats.hull(3)), ... % minx
                         -obj.feats.hull(2)/2, ... % miny
                         obj.feats.hull(3), ... % maxx
                         obj.feats.hull(2)/2 ]; % maxy
            for (f = 1:nw)
                p0r = rTu * ([obj.walls(f,1:2) 0 1].');
                p1r = rTu * ([obj.walls(f,3:4) 0 1].');
                p0in = obj.pointInHull(p0r,hulllims); 
                p1in = obj.pointInHull(p1r,hulllims);
                if p0in || p1in
                    c = 1;
                    i = f;
                    break;
                else % wall may cross the hull
                    x0 = p0r(1);
                    y0 = p0r(2);
                    x1 = p1r(1);
                    y1 = p1r(2);
                    
                    vx = x1-x0;
                    vy = y1-y0;
                    if (vy ~= 0)
                        l1 = (hulllims(2) - y0)/vy;
                        xi1 = x0 + l1 * vx;
                        l2 = (hulllims(4) - y0)/vy;
                        xi2 = x0 + l2 * vx;
                        if ((l1 >= 0) && (l1 <= 1) && (xi1 >= hulllims(1)) && (xi1 <= hulllims(3))) || ...
                           ((l2 >= 0) && (l2 <= 1) && (xi2 >= hulllims(1)) && (xi2 <= hulllims(3)))
                            c = 1;
                            i = f;
                            break;
                        end
                    end
                    if (vx ~= 0)
                        l1 = (hulllims(1) - x0)/vx;
                        yi1 = y0 + l1 * vy;
                        l2 = (hulllims(3) - x0)/vx;
                        yi2 = y0 + l2 * vy;
                        if ((l1 >= 0) && (l1 <= 1) && (yi1 >= hulllims(2)) && (yi1 <= hulllims(4))) || ...
                           ((l2 >= 0) && (l2 <= 1) && (yi2 >= hulllims(2)) && (yi2 <= hulllims(4)))
                            c = 1;
                            i = f;
                            break;
                        end
                    end                                       
                end
            end
        
        end
        
		function setMotor(obj,motor,power)        
		% Set the power of one motor. 
        % Note that this does not animate the robot drawing or simulate any
        % time increment in the system.
        % 
		% 	s.setMotor(motor,power);
		%
		% MOTOR is the motor, either 'L' or 'R'
		% POWER is the power, an integer within -100 and 100	
		
            which = Lego.checkMotor(motor);
            Lego.checkPower(power);
            obj.motors(which) = power;
        
		end
			
		function enc = readEncoder(obj,motor)
		% Read the encoder of one motor, as an integer value in degrees.
        % 
		% 	enc = s.readEncoder(motor);
		%
		% MOTOR is the motor, either 'L' or 'R'
        
            which = Lego.checkMotor(motor);
            enc = round(obj.encs(which));
		
		end
		
		function z = readUltrasonic(obj)
		% Return the last ultrasonic sensor that was simulated, as an integer
		% value in cm. If none has been simulated yet, or the sensor has not
		% detected any obstacle, return MAXSONAR.
        % 
		% 	z = s.readUltrasonic();
		%
        % This DOES NOT simulate a new sensor reading; just takes the last
        % one simulated with the simulate() method.

            z = Lego.MAXSONAR;
            if ~isnan(obj.lastsonar)
                z = round(obj.lastsonar * 100);
            end
   
		end
		
		function light = readLight(obj)
		% Return the last light sensor reading that was simulated, in the
		% range [0,MAXLIGHT], as an integer number. If none has been simulated 
		% yet, or there is no light map previously loaded, return MAXLIGHT.
        % 
		% 	light = s.readLight();
		%
        % This DOES NOT simulate a new sensor reading; just takes the last
        % one simulated with the simulate() method.
		
            light = Lego.MAXLIGHT;
            if ~isnan(obj.lastlight)
                light = round(obj.lastlight * Lego.MAXLIGHT);
            end
        
        end

  		function [angle,speed] = readGyro(obj)
		% Return the last gyro sensor readings that were simulated.
        % 
		% 	[angle,speed] = s.readGyro();
		%
        % ANGLE will be the angle in degrees (an integer), measured from
        % the last time the gyro was reset.
        % SPEED will be the angular speed in degrees per second (a real)
        % measured at the end of the last simulated time.
        % If there is no readings yet, return 0 and 0.0.
		% This DOES NOT simulate new sensor readings; just takes the last
        % ones simulated with the simulate() method.
		
            angle = 0;
            speed = 0;
            if ~isnan(obj.lastgyro(1))
                angle = round(rad2deg(obj.lastgyro(1)));
            end
            if ~isnan(obj.lastgyro(2))
                speed = round(rad2deg(obj.lastgyro(2)));
            end
        
        end

  		function resetGyro(obj)
		% Reset the measurements of the gyro sensor. The current angle
        % measured from the gyro is set to 0 at this simulation time.
        % 
		% 	s.resetGyro();

            obj.lastgyro = [NaN, NaN, obj.gt_pose(3)];
        
        end
        
        function [r,d,offsx,offsy,offsth,x,y,theta] = measureRobot(obj)
        % Simulate the manual measurement of different parameters of this
        % robot with a rule. They will have some error w.r.t. the real
        % parameters.
        % 
		% 	[r,d,offsx,offsy,offsth,x,y,theta] = s.measureRobot();
		%
        % Note that each time you call this method, different results are
        % obtained (for the same robot).
        %
        % R is the radius of the wheels, in meters, with a resolution of 1
        % mm.
        % D is the distance between wheels, in meters, with a resolution of
        % 1 mm.
        % OFFSX,OFFSY,OFFSTH are the offset of the sonar system as seen
        % from the robot system (in meters and degrees), with resolution of 
        % 1 mm in the first two parameters and 1 degree in the last one.
        % X,Y,THETA are the pose of the robot (m,rad) with 1 mm and 1
        % degree of resolution.
        
            r = round((obj.feats.r(randi(2)) + normrnd(0,0.003/2))*1000)/1000;
            d = round((obj.feats.d + normrnd(0,0.0075/2))*1000)/1000;
            
            offsx = round((obj.feats.rTs(1,4) + normrnd(0,0.003/2))*1000)/1000;
            offsy = round((obj.feats.rTs(2,4) + normrnd(0,0.003/2))*1000)/1000;
            offsth = round(rad2deg(atan2(obj.feats.rTs(2,1),obj.feats.rTs(1,1)) + normrnd(0,deg2rad(5)/2)));
            
            x = round((obj.gt_pose(1) + normrnd(0,0.015/2))*1000)/1000;
            y = round((obj.gt_pose(2) + normrnd(0,0.015/2))*1000)/1000;
            theta = deg2rad(round(rad2deg(obj.gt_pose(3) + normrnd(0,deg2rad(5)/2))));
            
        end
        
        
        function st = getSimTime(obj)
        % Return the current simulation time in seconds.
        %
        %	st = s.getSimTime();
        
            st = obj.t;
        
        end

        function simulate(obj,inct)
        % Simulate the system during the next INCT seconds with 100
        % intermediate steps. Afterwards, redraw the scenario. 
        % It also simulates a new reading of the sonar and of the light sensor 
        % after that increment of time.
        %
        %	s.simulate(inct);
        % 
        % Note that this method does not check collisions; use the
        % collision() method after calling this one.
        %
        % INCT is the time to simulate from now.
        
            Lego.checkIsPositive(inct,'INCT');
            
            oldstate = struct('gtpose',obj.gt_pose.',... 
                                'encl',obj.encs(1),... 
                                'encr',obj.encs(2),... 
                                'wl',obj.alphas(1),...
                                'wr',obj.alphas(2),...
                                'al',obj.angs(1),...
                                'ar',obj.angs(1),...
                                't',obj.t...
                                );
            % simulate motion for 100 steps within INCT
            newstate = simulationstep(obj.motors(1),obj.motors(2),...
                                      inct,oldstate,...
                                      obj.feats.d,...
                                      obj.feats.r(1),obj.feats.r(2));
            obj.gt_pose = newstate.gtpose.';
            obj.encs = [newstate.encl newstate.encr];
            obj.alphas = [newstate.wl newstate.wr];
            obj.angs = [newstate.al newstate.ar];
            obj.t = newstate.t;

            % simulate the sonar
            uTr = obj.robot2univMatrix();
            pos = uTr * obj.feats.rTs * [0;0;0;1];
            posx = uTr * obj.feats.rTs * [1;0;0;1];
            thetasonaru = atan2(posx(2)-pos(2),posx(1)-pos(1)); % in -pi...+pi
            obj.lastsonar = simulationsonar(pos(1),pos(2),thetasonaru,...
                                obj.walls,...
                                obj.feats.semiangle_sonar,...
                                obj.feats.sigmas_sonar(1),...
                                obj.feats.sigmas_sonar(2),...
                                Lego.MAXSONAR/100);
            if (obj.lastsonar >= Lego.MAXSONAR/100 - 1e-6)
                obj.lastsonar = NaN;
            end
                            
            % simulate the light sensor
            pl = uTr * obj.feats.rTl * [0;0;0;1];
            obj.lastlight = simulationlightsensor(...
                                      squeeze(obj.light.lightmap(:,:,1)),...
                                      obj.light.xul,obj.light.yul,...
                                      obj.light.wpix,obj.light.hpix,...
                                      pl(1),pl(2),...
                                      obj.feats.radius_light,...
                                      obj.feats.sigma_light);
                
            obj.drawAll();
            
            % simulate the gyro sensor
            obj.lastgyro(2) = newstate.wend + normrnd(0,deg2rad(6.7/2)); % w
            incor = - ( newstate.gtpose(3) - obj.lastgyro(3) ); % non-same-sign-as-w
            % --- mode I): gaussian noise in each increment; moving ref
            sigmaincor = abs(incor * (deg2rad(45/2) / (pi/2)));
            if isnan(obj.lastgyro(1))
               obj.lastgyro(1) = incor + normrnd(0,sigmaincor); % angle (rad)
            else
               obj.lastgyro(1) = obj.lastgyro(1) + incor + normrnd(0,sigmaincor); % angle (rad)
            end
            obj.lastgyro(3) = newstate.gtpose(3);
            % --- mode II): gaussian noise w.r.t. last reset
            % sigmaincor = abs(incor * (deg2rad(3/2) / (pi/2)));
            % obj.lastgyro(1) = incor + normrnd(0,sigmaincor); % angle (rad)
        
        end
        
        function h = getFigure(obj)
        % Return the handler of the figure where the robot is being drawn.
        %
        %	h = s.getFigure();
        
            h = obj.disp;
        
        end
        
        function redraw(obj)
        % Redraw the simulator figure.
        %
        %	s.redraw();
        
            obj.drawAll();
        
        end
        
        function changeDrawEnv(obj,area,equalaxes)
        % Change the viewport of the simulator to one fixed on the plane.
        %
        %	s.changeDrawEnv(area,equalaxes);
        %
        % AREA is a vector with minx, miny, maxx, maxy
        % If EQUALAXES == 1, adjust the area to include the desired one but
        % having aspect ratio == 1.
        
            Lego.checkIsVectorOfN(area,4,'area');
            incx = area(3) - area(1);
            incy = area(4) - area(2);
            if (incx <= 0) || (incy <= 0)
                error('Invalid viewport area');
            end
            obj.environ = struct('mode',0,...
                                 'minx',area(1),...
                                 'miny',area(2),...
                                 'maxx',area(3),...
                                 'maxy',area(4));
            if (equalaxes)
                obj.adjustAspect();
            end
        
        end

        function changeDrawEnvMove(obj,radii,equalaxes)
        % Change the viewport of the simulator to one that moves along the
        % robot.
        %
        %	s.changeDrawEnvMove(radii,equalaxes);
        %
        % RADII is the width of the viewport in the x axis (1) and in the y
        % axis (2) that will be shown.
        % If EQUALAXES == 1, adjust the area to include the desired one but
        % having aspect ratio == 1.

            Lego.checkIsVectorOfN(radii,2,'radii');
            if (radii(1) <= 0) || (radii(2) <= 0)
                error('Invalid viewport area');
            end
            obj.environ = struct('mode',1,...
                                 'radiusx',radii(1),...
                                 'radiusy',radii(2));
            if equalaxes
                obj.adjustAspect();
            end
        
        end

        function setUltrasonicDrawing(obj,ed)
        % Enable (ED == 1) or disable (ED == 0) the drawings of the
        % ultrasonic data. If ED == 2, enable it and also draws the beam.
        %
        %	s.setUltrasonicDrawing(ed);
        
            Lego.checkIsNumScalar(ed,'ed');
            obj.drawsonar = ed;
        
        end
        
        function setLightDrawing(obj,ed)
        % Enable (ED == 1) or disable (ED == 0) the drawings of the
        % light sensor device and of the light map data. 
        %
        %	s.setLightDrawing(ed);
        
            Lego.checkIsNumScalar(ed,'ed');
            obj.drawlight = ed;
        
        end
        
    end % --- end public methods
    

    methods (Static, Access = private)

        function unimplemented()
        % produce an unimplemented error
            error('Unimplemented operation');
        end

        function checkNonEmpty(v,txt)
            if (isempty(v))
                error('%s must be non-empty',txt);
            end
        end
        
        function checkIsNumScalar(v,txt)
            if (~isscalar(v))
                error('%s must be a scalar',txt)
            end
            if (ischar(v))
                error('%s must be a numeric scalar',txt);
            end
        end
        
        function checkIsVectorOfN(v,n,txt)
            if length(v) ~= n
                error('%s must have %d elements',txt,n);
            end
        end
        
        function checkIsHT(t,txt)
            [r,c] = size(t);
            if (r ~= 4) || (c ~= 4)
                error('%s is not a 4x4 matrix',txt);
            end
        end
        
        function checkIsPositive(v,txt)
            Lego.checkNonEmpty(v,txt);
            Lego.checkIsNumScalar(v,txt);
            if (v <= 0)
                error('%s must be positive',txt)
            end
        end
        
        function checkPower(v)
            Lego.checkNonEmpty(v,'Power ');
            Lego.checkIsNumScalar(v,'Power ');
            if (mod(v,1) ~= 0)
                error('Power must be an integer');
            end
            if (v < -100) || (v > 100)
                error('Power must be in [-100,+100]');
            end
        end
        
        function which = checkMotor(v)
            if v == 'L'
                which = 1;
            elseif v == 'R'
                which = 2;
            else
                error('Invalid motor name');
            end            
        end
        
        function in = pointInHull(p,hull) % hull is minx,miny,maxx,maxy
            if (p(1) > hull(1)) && (p(1) < hull(3)) && ...obj.lastlight * obj.MAXLIGHT
               (p(2) > hull(2)) && (p(2) < hull(4))
                in = 1;
            else
                in = 0;
            end
        end
        
    end
    
    
    methods (Access = private)
        
        function figexists = checkFigure(obj)
            
            figexists = (findobj(obj.disp,'type','figure') == obj.disp);
            
        end
        
        function adjustAspect(obj)

            switch obj.environ.mode
                case 0
                    incx = obj.environ.maxx - obj.environ.minx;
                    incy = obj.environ.maxy - obj.environ.miny;
                case 1
                    incx = obj.environ.radiusx;
                    incy = obj.environ.radiusy;
                otherwise
                    error('Invalid viewport mode');
            end
            switch obj.environ.mode
                case 0
                    ar = incy/incx;
                    if ar > 1
                        xneeded = incy-incx;
                        obj.environ.minx = obj.environ.minx - xneeded/2;
                        obj.environ.maxx = obj.environ.maxx + xneeded/2;
                    else
                        yneeded = incx-incy;
                        obj.environ.miny = obj.environ.miny - yneeded/2;
                        obj.environ.maxy = obj.environ.maxy + yneeded/2;
                    end
                case 1
                    m = max([obj.environ.radiusx,obj.environ.radiusy]);
                    obj.environ.radiusx = m;
                    obj.environ.radiusy = m;
            end
            
        end
        
        function updateRobShape(obj)
            
            lx = obj.feats.hull(1);
            ly = obj.feats.hull(2);
            lf = obj.feats.hull(3);
            lxmlf = lx - lf;
            lyd2 = ly/2;
            lo = min(obj.feats.hull(1:2))/2;
            obj.robshape = [-lxmlf,lyd2; ...
                            lf,lyd2;...
                            lf,0;...
                            lf+lo,0;...
                            lf,0;...
                            lf,-lyd2;...
                            -lxmlf,-lyd2;...
                            -lxmlf,lyd2];            
                        
        end
        
        function uTr = robot2univMatrix(obj)
            
            c = cos(obj.gt_pose(3));
            s = sin(obj.gt_pose(3));
            uTr = [ c -s 0 obj.gt_pose(1) ; ...
                    s c  0 obj.gt_pose(2) ; ...
                    0 0  1 0 ; ...
                    0 0  0 1 ]; % uTr
                
        end
               
        function drawAll(obj)

            if ~obj.checkFigure()
                fprintf('Simulation figure re-created.\n');
                obj.disp = figure;
            end
            
            % calculations for drawing
            pss = [0 -0.015 0 ; -0.05 0 0 ; 0 0.015 0].'; % sonar shape
            pssr = [ obj.feats.rTs * [ pss(:,1) ; 1] , ...
                     obj.feats.rTs * [ pss(:,2) ; 1] , ...
                     obj.feats.rTs * [ pss(:,3) ; 1] ];
            c = cos(obj.gt_pose(3));
            s = sin(obj.gt_pose(3));
            uTr = obj.robot2univMatrix();
            pssu = [ uTr * pssr(:,1) , ...
                     uTr * pssr(:,2) , ...
                     uTr * pssr(:,3) ];
            cs = [0 0 0 1].'; %origin of sonar
            csr = obj.feats.rTs * cs;
            csu = uTr * csr;
            if (obj.drawsonar) && (~isnan(obj.lastsonar))
                z = obj.readUltrasonic() / 100;
                sonarpoint = uTr * obj.feats.rTs * [z;0;0;1];
            else
                z = NaN;
            end
            plu = uTr * obj.feats.rTl * [0;0;0;1];

            % start drawing
            figure(obj.disp);
            clf;
            hold on;
            
            % lightmap
            if obj.drawlight
                
                if ~isempty(obj.light.lightmap)
                    [h,w,~] = size(obj.light.lightmap);
                    image([obj.light.xul obj.light.xul + w*obj.light.wpix],...
                          [obj.light.yul obj.light.yul - h*obj.light.hpix],...
                          obj.light.lightmap);
                    % the upper-left pixel is drawn with its center at
                    % xul,yul
                end
            end
            
            % draw univ system
            plot([0 1],[0 0],'r-');
            plot([0 0],[0 1],'y-');
            axis equal;
            
            switch (obj.environ.mode)
                case 0 % fixed
                    xlim([obj.environ.minx obj.environ.maxx]);
                    ylim([obj.environ.miny obj.environ.maxy]);
                case 1 % moving
                    xlim(obj.gt_pose(1) + obj.environ.radiusx * [-1,1]);
                    ylim(obj.gt_pose(2) + obj.environ.radiusy * [-1,1]);
                otherwise
                    error('Invalid viewport mode');
            end
            
            % draw walls
            if ~isempty(obj.walls)
                [nw,~] = size(obj.walls);
                for (f = 1:nw)
                    plot([obj.walls(f,1) obj.walls(f,3)],...
                         [obj.walls(f,2) obj.walls(f,4)],...
                         'k-','LineWidth',3);
                end
            end
            
            % draw robot
            colorrob = 'bw'; % border and fill
            simulationdrawrobot(obj.disp,uTr,obj.gt_pose(3),colorrob,'shape',obj.robshape); % body
            plot([pssu(1,:) pssu(1,1)],[pssu(2,:) pssu(2,1)],sprintf('%c-',colorrob(1))); % sonar
            plot(csu(1),csu(2),sprintf('%c.',colorrob(1)));
            plot(plu(1),plu(2),sprintf('%co',colorrob(1)),'MarkerSize',6);
                      
            % draw sonar data
            if ~isnan(z)
                if (obj.drawsonar == 2)
                    plot([csu(1) sonarpoint(1)],[csu(2) sonarpoint(2)],'g-');
                else
                    plot(sonarpoint(1),sonarpoint(2),'g*');
                end
            end           

            grid;
            title(sprintf('[LM^3L^T v%s]  t=%.3fs  s=%dcm  l=%d  g=%d^{o}',...
                          Lego.VER,...
                          obj.t,...
                          obj.readUltrasonic(),...
                          obj.readLight(),...
                          obj.readGyro()));
            drawnow;
            
        end
                    
    end

    
end
