/*---------------------------------------------------------------
	FILE: CActionCollection.h
	USE: See above

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef CActionCollection_H
#define CActionCollection_H

#include <MRPT/MRML/CAction.h>
#include <MRPT/MRML/CActionRobotMovement2D.h>
#include <MRPT/UTILS/CSerializable.h>

namespace MRML
{
	/** Declares a class for storing a collection of robot actions. It is used in MRML::CActionObservationSequence,
	 *    for logs storage and particle filter based simulations.
     *
	 * \sa CAction, CActionObservationSequence
	 */
	class CActionCollection : public CSerializable
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CActionCollection )

	protected:
		/** The actions:
		  */
		std::deque<CAction*>	actions;

	 public:
		 /** Constructor
		   */
		CActionCollection();

		/** Constructor from a single action.
		   */
		CActionCollection( CAction &a );

		/** Copy Constructor
		   */
		CActionCollection(const CActionCollection &o );

		/** Copy operator
		  */
		void  operator = (const CActionCollection &o );

		/** Destructor
		   */
		~CActionCollection();

		/** Erase all actions from the list.
		  */
		void  clear();

		/** Access the i'th action.DO NOT MODIFY the returned object, make a copy of ir with "CSerializable::duplicate" if desired.
		  *  First element is 0.
		  * \exception std::exception On index out of bounds.
		  */
		CAction	*  get(unsigned int index);

		/** Access the first action found with a desired class. Pass the desired class with CLASS_ID( classname )
		  *   DO NOT MODIFY the returned object, make a copy of ir with "CSerializable::duplicate" if desired.
		  * \returns The action, or NULL if not found.
		  */
		CAction	*  getByClass( UTILS::TRuntimeClassId	*desired_class );

		/** Add a new object to the list.
		  */
		void  insert(CAction	&action);

		/** Returns the actions count in the collection.
		  */
		size_t  size();

		/** Returns the best pose increment estimator in the collection, based on the determinant of its pose change covariance matrix.
		  * \return The estimation, or NULL if none is available.
		  */
		const CActionRobotMovement2D	*  getBestMovementEstimation() const;

		/** Returns the pose increment estimator in the collection having the specified type.
		  * \return The estimation, or NULL if none is available.
		  */
		CActionRobotMovement2D	*  getMovementEstimationByType( CActionRobotMovement2D::TEstimationMethod method);

		/** Remove an action from the list by its index.
		  * \exception std::exception On index out of bounds.
		  */
		void  eraseByIndex(unsigned int index);


	}; // End of class def.

} // End of namespace

#endif
