/*---------------------------------------------------------------
	FILE: utils_defs.h
	USE: Declares some useful constants and small "defines"

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef UTILSDEFS_H
#define UTILSDEFS_H

#include <MRPT/config.h>
#include <MRPT/UTILS/MRPT_OS.h>

/* ------------------------------------
          Disable some warnings
   ------------------------------------ */
#if defined(_MSC_VER)
	// (Compiler: Visual C++) Disable warning for too long debug names:
	#pragma warning(disable:4786)
	// (Compiler: Visual C++) Disable warning for unreachable code (I don't know why some of these errors appear in the STANDARD LIBRARY headers with Visual Studio 2003!):
	#pragma warning(disable:4702)
#endif

#if defined(__BORLANDC__)
	// (Compiler: Borland C++) Disable a warning for inline functions
	#pragma warn -8027
        // (Compiler: Borland C++) Disable a warning for override of virtual functions, while in reality there are many virtual function with different parameters
	#pragma warn -8012
	#pragma warn -8022
#endif

// For the new more secure library in VC++8
#if defined(_MSC_VER) && (_MSC_VER>=1400)
	#if !defined(_CRT_SECURE_CPP_OVERLOAD_STANDARD_NAMES)
		#define _CRT_SECURE_CPP_OVERLOAD_STANDARD_NAMES 1
	#endif
#endif

/* ------------------------------------------------
    Automatically tell the linker to include
	 the correct MRPT ".lib" file into the
	 user's application.
   ----------------------------------------------- */
// Borland C++
#if defined(__BORLANDC__)
	#if defined(_DEBUG)
		#pragma comment (lib,"libMRPTd_bcc.lib")
	#else
		#pragma comment (lib,"libMRPT_bcc.lib")
	#endif
#endif

// Microsoft Visual C++ 6.0
#if defined(_MSC_VER) && (_MSC_VER==1200)
	#if defined(_DEBUG)
		#pragma comment (lib,"libMRPTd_MSVC6.lib")
	#else
		#pragma comment (lib,"libMRPT_MSVC6.lib")
	#endif
#endif

// Microsoft Visual C++ 2003
#if defined(_MSC_VER) && ((_MSC_VER==1300) || (_MSC_VER==1310))
	#if defined(_DEBUG)
		#pragma comment (lib,"libMRPTd_MSVC7.lib")
	#else
		#pragma comment (lib,"libMRPT_MSVC7.lib")
	#endif
#endif

// Microsoft Visual C++ 2005
#if defined(_MSC_VER) && (_MSC_VER==1400)
	#if defined(_DEBUG)
		#pragma comment (lib,"libMRPTd_MSVC8.lib")
	#else
		#pragma comment (lib,"libMRPT_MSVC8.lib")
	#endif
#endif

// A cross-compiler definition for "deprecated"-warnings
#if defined(__GNUC__) && (__GNUC__ - 0 > 3 || (__GNUC__ - 0 == 3 && __GNUC_MINOR__ - 0 >= 2))
   /* gcc >= 3.2 */
#   define MRPT_DEPRECATED_PRE
#   define MRPT_DEPRECATED_POST __attribute__ ((deprecated))
# elif defined(_MSC_VER) && (_MSC_VER >= 1300)
  /* msvc >= 7 */
#   define MRPT_DEPRECATED_PRE  __declspec(deprecated)
#   define MRPT_DEPRECATED_POST
# else
#  define MRPT_DEPRECATED_PRE
#  define MRPT_DEPRECATED_POST
# endif


// Utils C++ headers:
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <float.h>
#include <stdarg.h>
#include <memory.h>
#include <string.h>

// C++ STD Library:
#include <algorithm>
#include <fstream>
#include <iostream>
#include <vector>
#include <deque>
#include <queue>
#include <map>
#include <set>
#include <string>
#include <exception>
#include <stdexcept>
#include <limits>
#include <iomanip>
#include <locale>
#include <sstream>
#include <cassert>

// The  modifier is not defined on GCC
#if defined(__GNUC__)
	#include <stdint.h>
#endif

#if defined(__BORLANDC__) || defined(_MSC_VER)
   typedef signed __int64 int64_t;
   typedef unsigned __int64 uint64_t;

   typedef signed __int32 int32_t;
   typedef unsigned __int32 uint32_t;

   typedef signed __int16 int16_t;
   typedef unsigned __int16 uint16_t;

   typedef signed __int8 int8_t;
   typedef unsigned __int8 uint8_t;
#endif


namespace UTILS
{
	//  Defintions of data types:
	// ---------------------------------------------------------
	/** A type for passing a vector of longs.
	*/
	typedef std::vector<long> vector_long;

	/** A type for passing a vector of ints.
	*/
	typedef std::vector<int> vector_int;

	/** A type for passing a vector of floats.
	*/
	typedef std::vector<float> vector_float;

	/** A type for passing a vector of doubles.
	*/
	typedef std::vector<double> vector_double;

	/** A type for passing a vector of 16-bits unsigned numbers.
	*/
	typedef std::vector<unsigned short> vector_word;

	/** A type for passing a vector of 32-bits unsigned numbers.
	*/
	typedef std::vector<unsigned int> vector_uint;

	/** A type for passing a vector of strings.
	*/
	typedef std::vector<std::string> vector_string;

	/** A type for passing a vector of boolean values.
	*/
	typedef std::vector<bool> vector_bool;

   /** A type for passing a vector of bytes.
	*/
	typedef std::vector<uint8_t> vector_byte;

	// Forward definition: (Required for Visual C++ 6 implementation of THROW_EXCEPTION...)
	namespace SystemUtils
	{
		std::string  extractFileName(const std::string &filePath);
	}

} // End of namespace


using namespace UTILS;

/** Forward definitions for numerical test functions:
  */
namespace UTILS
{
    class CFileStream;

	namespace MATH
	{
	    bool isNan(float v);
	    bool isNan(double v);
	    bool isFinite(float v);
	    bool isFinite(double v);
	}
}

/** A macro for obtaining the name of the current function:
  */
#if defined(__BORLANDC__)
		#define	__CURRENT_FUNCTION_NAME__	__FUNC__
#elif defined(_MSC_VER) && (_MSC_VER>=1300)
		#define	__CURRENT_FUNCTION_NAME__	__FUNCTION__
#elif defined(_MSC_VER) && (_MSC_VER<1300)
		// Visual C++ 6 HAS NOT A __FUNCTION__ equivalent.
#define	__CURRENT_FUNCTION_NAME__	::UTILS::SystemUtils::extractFileName(__FILE__).c_str()
#else
		#define	__CURRENT_FUNCTION_NAME__	__PRETTY_FUNCTION__
#endif


/** \def THROW_EXCEPTION(msg)
 * Defines a unified way of reporting exceptions
 * \sa MRPT_TRY_START, MRPT_TRY_END, THROW_EXCEPTION_CUSTOM_MSG1
 */
#define THROW_EXCEPTION(msg)	\
	{\
		char aux[1000];\
		::UTILS::MRPT_OS::sprintf(aux,1000,"\n\n =============== MRPT EXCEPTION =============\n%s,line %u:\n ",\
			__CURRENT_FUNCTION_NAME__,\
			__LINE__); \
		std::string auxCompStr( aux ); \
		auxCompStr+= msg; \
		auxCompStr+= std::string("\nCALL STACK:\n"); \
		throw std::logic_error( auxCompStr );\
	}\

/** \def THROW_EXCEPTION_CUSTOM_MSG1
  * \param e The caught exception.
  *	\param msg Is a char* or std::string.
  */
#define THROW_EXCEPTION_CUSTOM_MSG1(msg,param1)	\
	{\
		char aux[1000],auxUsr[2000];\
		::UTILS::MRPT_OS::sprintf(aux,1000,"\n\n =============== MRPT EXCEPTION =============\n%s,line %u:\n ",\
			__CURRENT_FUNCTION_NAME__,\
			__LINE__); \
		::UTILS::MRPT_OS::sprintf(auxUsr,sizeof(auxUsr), msg,param1 );\
		std::string auxCompStr( aux ); \
		auxCompStr+= auxUsr; \
		auxCompStr+= std::string("\nCALL STACK:\n"); \
		throw std::logic_error( auxCompStr );\
	}\


/** \def THROW_TYPED_EXCEPTION(msg,exceptionClass)
 * Defines a unified way of reporting exceptions of type different from "std::exception"
 * \sa MRPT_TRY_START, MRPT_TRY_END
 */
#define THROW_TYPED_EXCEPTION(msg,exceptionClass)	\
	{\
		char aux[1000];\
		::UTILS::MRPT_OS::sprintf(aux,1000,"\n\n =============== MRPT EXCEPTION =============\n%s,line %u:\n ",\
			__CURRENT_FUNCTION_NAME__,\
			__LINE__); \
		std::string auxCompStr( aux ); \
		auxCompStr+= msg; \
		auxCompStr+= std::string("\nCALL STACK:\n"); \
		throw exceptionClass( auxCompStr );\
	}\

/** \def THROW_STACKED_EXCEPTION
 * \sa MRPT_TRY_START, MRPT_TRY_END
 */
#define THROW_STACKED_EXCEPTION(e)	\
	{\
		char aux[1000];\
		::UTILS::MRPT_OS::sprintf(aux,sizeof(aux)," %s, line %u\n",\
			__CURRENT_FUNCTION_NAME__,\
			__LINE__);\
		std::string auxCompStr( e.what() ); \
		auxCompStr+= aux; \
		throw std::logic_error( auxCompStr );\
	}\

/** \def THROW_STACKED_EXCEPTION_CUSTOM_MSG
  * \param e The caught exception.
  *	\param msg Is a char* or std::string.
  */
#define THROW_STACKED_EXCEPTION_CUSTOM_MSG1(e,msg)	\
	{\
		char aux[1000];\
		::UTILS::MRPT_OS::sprintf(aux,sizeof(aux)," %s, line %u (User msg above)\n",\
			__CURRENT_FUNCTION_NAME__,\
			__LINE__);\
		std::string auxCompStr( e.what() ); \
		auxCompStr+= msg; \
		throw std::logic_error( auxCompStr );\
	}\

/** \def THROW_STACKED_EXCEPTION_CUSTOM_MSG
  * \param e The caught exception.
  *	\param stuff Is a printf-like sequence of params, e.g: "The error happens for x=%i",x
  */
#define THROW_STACKED_EXCEPTION_CUSTOM_MSG2(e,stuff,param1)	\
	{\
		char aux[1000],auxUsr[2000];\
		::UTILS::MRPT_OS::sprintf(aux,sizeof(aux)," %s, line %u (User msg above)\n",\
			__CURRENT_FUNCTION_NAME__,\
			__LINE__);\
		::UTILS::MRPT_OS::sprintf(auxUsr,sizeof(auxUsr), stuff,param1 );\
		std::string auxCompStr( e.what() ); \
		auxCompStr+= auxUsr; \
		auxCompStr+= aux; \
		throw std::logic_error( auxCompStr );\
	}\

/**
 * Defines an assertion mechanism.
 * \sa MRPT_TRY_START, MRPT_TRY_END
 */
#define ASSERT_(f) \
	{ \
	if (!(f)) \
		{ \
			::std::string tmpStr("Assert condition failed: "); \
			tmpStr+= #f; \
			THROW_EXCEPTION( tmpStr.c_str() ); \
		} \
	} \

/** Throws an exception if the number is NaN, IND, or +/-INF, or return the same number otherwise.
  * \todo Add support for isnanf, etc.
  */
#define MRPT_CHECK_NORMAL_NUMBER(val) \
	{ \
		double	v=(val); \
		if (UTILS::MATH::isNan(v)) THROW_EXCEPTION("Check failed (value is NaN)"); \
		if (!UTILS::MATH::isFinite(v)) THROW_EXCEPTION("Check failed (value is not finite)"); \
	} \


/** Can be used to avoid "not used parameters" warnings from the compiler
 */
#define MRPT_UNUSED_PARAM(a)		(void)(a)


/** The start of a standard MRPT "try...catch()" block that allows tracing throw the call stack after an exception.
  * \sa MRPT_TRY_END,MRPT_TRY_END_WITH_CLEAN_UP
  */
#define MRPT_TRY_START	\
	try { \


/** The end of a standard MRPT "try...catch()" block that allows tracing throw the call stack after an exception.
  * \sa MRPT_TRY_START,MRPT_TRY_END_WITH_CLEAN_UP
  */
#define MRPT_TRY_END	\
	} \
	catch (std::exception &e) \
	{ \
		THROW_STACKED_EXCEPTION(e); \
	} \
	catch (...) \
	{ \
		THROW_EXCEPTION("Unexpected runtime error!"); \
	} \

/** The end of a standard MRPT "try...catch()" block that allows tracing throw the call stack after an exception, including a "clean up" piece of code to be run before throwing the exceptions.
  * \sa MRPT_TRY_END,MRPT_TRY_START
  */
#define MRPT_TRY_END_WITH_CLEAN_UP(stuff)	\
	} \
	catch (std::exception &e) \
	{ \
		{stuff} \
		THROW_STACKED_EXCEPTION(e); \
	} \
	catch (...) \
	{ \
		{ stuff } \
		THROW_EXCEPTION("Unexpected runtime error!"); \
	} \

// Generic constants and defines:
// ---------------------------------------------------------

#ifndef M_PI
/** The PI constant
 */
	#define M_PI 3.1415926535897932384626433832795
#endif

#ifndef M_2PI
/** The 2*PI constant
 */
	#define M_2PI 6.283185307179586476925286766559
#endif

#define M_PIf  3.14159265358979f
#define M_2PIf 6.28318530717959f

/** Degrees to radians
 */
__inline double DEG2RAD(const double &x)
{
	return x*M_PI/180.0;
}
/** Degrees to radians
 */
__inline float DEG2RAD(const float &x)
{
	return x*M_PIf/180.0f;
}
/** Degrees to radians
 */
__inline float DEG2RAD(const int &x)
{
	return x*M_PIf/180.0f;
}

/** Radians to degrees
 */
__inline double RAD2DEG(const double &x)
{
	return x*180.0/M_PI;
}
/** Radians to degrees
 */
__inline float RAD2DEG(const float &x)
{
	return x*180.0f/M_PIf;
}

/** Returns the sign of X as "1" or "-1"
 */
#define sign(X)		(((X)<0) ? (-1):(1))

/** Returns the closer integer (int) to x
 */
#define round(X)	((int)floor(X+0.5))

/** Inline function for the square of a number.
 */
template<class T>
inline T square(const T& x)    { return x*x; }

/** Inline function for the absolute value of any real number type.
 */
//inline float abs (float v) { return (float)fabs( v); }
/** Inline function for the absolute value of any real number type.
 */
//inline double abs (double v) { return fabs( v); }
/** Inline function for the absolute value of any real number type.
 */
//inline long double abs (long double v) { return fabsl( v); }

// Min & Max:
#ifndef min
/** Returns the smallest of two numbers
 */
	#define min(A,B)	((A)<(B) ? (A):(B))
#endif

#ifndef max
/** Returns the biggest of two numbers
 */
	#define max(A,B)	((A)>(B) ? (A):(B))
#endif

#define max3(a,b,c)	((a) > (b) ? max((a),(c)) : max((b),(c)))
#define min3(a,b,c)	((a) < (b) ? min((a),(c)) : min((b),(c)))

/** Generic template for swaping a pair of variables
  */
template <class T> void swap(T &a, T&b)
{
	T	temp(a);
	(a)=(b);
	(b)=(temp);
}

namespace UTILS
{
	// --------------------------
	// Functors:
	//  Ret: NO  In:1-3
	// --------------------------

	/** A generic functor type for functions accepting 1 input arguments and returning nothing.
	*/
	typedef void (*TFunctor_noRet_1inputs)(const void *);

	/** A generic functor type for functions accepting 2 input arguments and returning nothing.
	*/
	typedef void (*TFunctor_noRet_2inputs)(const void *,const void *);

	/** A generic functor type for functions accepting 3 input arguments and returning nothing.
	*/
	typedef void (*TFunctor_noRet_3inputs)(const void *,const void *,const void *);

	// --------------------------
	// Functors:
	//  Ret: double  In:1-3
	// --------------------------

	/** A generic functor type for functions accepting 1 input arguments and returning a double value.
	*/
	typedef double (*TFunctor_retDouble_1inputs)(const void *);

	/** A generic functor type for functions accepting 2 input arguments and returning a double value.
	*/
	typedef double (*TFunctor_retDouble_2inputs)(const void *,const void *);

	/** A generic functor type for functions accepting 3 input arguments and returning a double value.
	*/
	typedef double (*TFunctor_retDouble_3inputs)(const void *,const void *,const void *);

	// --------------------------
	// Functors:
	//  Ret: vector  In: vector
	// --------------------------

	/** A generic functor type for functions accepting 1 vector and returning 1 vector.
	*/
	typedef void (*TFunctor_retVecDbl_inpVecDbl)(const vector_double &in, vector_double &out);

	/** A generic functor type for functions accepting 1 vector and returning 1 vector.
	*/
	typedef void (*TFunctor_retVecFlt_inpVecFlt)(const vector_float &in, vector_float &out);

	/** A generic functor type for functions accepting 1 vector and returning 1 vector.
	*/
	typedef void (*TFunctor_retVecInt_inpVecInt)(const vector_int &in, vector_int &out);

	// --------------------------
	// Functors:
	//  Ret: double  In: vectors
	// --------------------------

	/** A generic functor type for functions accepting 1 vector and returning 1 double.
	*/
	typedef double (*TFunctor_retDbl_inp1VecDbl)(const vector_double &in1);

	/** A generic functor type for functions accepting 2 vectors and returning 1 double.
	*/
	typedef double (*TFunctor_retDbl_inp2VecDbl)(const vector_double &in1,const vector_double &in2);

	/** A generic functor type for functions accepting 3 vectors and returning 1 double.
	*/
	typedef double (*TFunctor_retDbl_inp3VecDbl)(const vector_double &in1,const vector_double &in2,const vector_double &in3);


	/** For performing type casting from a pointer to its numeric value.
	*/
	#if defined(_MSC_VER) && (_MSC_VER>=1300)
		typedef unsigned long long POINTER_TYPE;
	#else
		typedef unsigned long POINTER_TYPE;
	#endif


	/** A template function for printing out the contents of a std::vector variable.
		*/
	template <class T>
	std::ostream& operator << (std::ostream& out, const std::vector<T> &d)
	{
		out << "[";
		for (typename std::vector<T>::const_iterator it = d.begin();it!=d.end();it++)
		{
			out << *it;
			if (it!=d.end()-1)
				out << ",";
		}
		out << "]";
		return out;
	}

	/** A template function for printing out the contents of a std::vector variable.
		*/
	template <class T>
	std::ostream& operator << (std::ostream& out, std::vector<T> *d)
	{
		out << "[";
		for (typename std::vector<T>::const_iterator it = d->begin();it!=d->end();it++)
		{
			out << *it;
			if (it!=d->end()-1)
				out << ",";
		}
		out << "]";
		return out;
	}

	/** Loads one row of a text file as a numerical std::vector.
	  * \return false on EOF or invalid format.
	  * The body of the function is implemented in MATH.cpp
		*/
	bool loadVector( CFileStream &f, std::vector<int> &d);

	/** Loads one row of a text file as a numerical std::vector.
	  * \return false on EOF or invalid format.
	  * The body of the function is implemented in MATH.cpp
		*/
	bool loadVector( CFileStream &f, std::vector<double> &d);


	/** A template function for the operator + between a scalar and a std::vector.
		*/
	template <class T>
	std::vector<T>  operator + (const std::vector<T> &a, T b)
	{
		typename std::vector<T>	res(a.size());
		typename std::vector<T>::const_iterator	it_a;
		typename std::vector<T>::iterator		it_res;

		for (it_a=a.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_res++)
			*it_res = *it_a + b;

		return res;
	}

	/** A template function for the operator - between a scalar and a std::vector.
		*/
	template <class T>
	std::vector<T>  operator - (const std::vector<T> &a, T b)
	{
		typename std::vector<T>	res(a.size());
		typename std::vector<T>::const_iterator	it_a;
		typename std::vector<T>::iterator		it_res;

		for (it_a=a.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_res++)
			*it_res = *it_a - b;

		return res;
	}

	/** A template function for the operator * between a scalar and a std::vector.
		*/
	template <class T>
	std::vector<T>  operator * (const std::vector<T> &a, T b)
	{
		typename std::vector<T>	res(a.size());
		typename std::vector<T>::const_iterator	it_a;
		typename std::vector<T>::iterator		it_res;

		for (it_a=a.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_res++)
			*it_res = *it_a * b;

		return res;
	}

	/** A template function for dividing a std::vector by a scalar number.
		*/
	template <class T>
	std::vector<T>  operator / (const std::vector<T> &a, T b)
	{
		typename std::vector<T>	res(a.size());
		typename std::vector<T>::const_iterator	it_a;
		typename std::vector<T>::iterator		it_res;

		if (b==0)
			THROW_EXCEPTION("Division by zero: execution aborted!");

		for (it_a=a.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_res++)
			*it_res = *it_a / b;

		return res;
	}

	/** A template function for adding std::vector's having the same number of elements.
		*/
	template <class T>
	std::vector<T>  operator + (const std::vector<T> &a, const std::vector<T> &b)
	{
		ASSERT_(a.size()==b.size());

		typename std::vector<T>	res(a.size());
		typename std::vector<T>::const_iterator	it_a,it_b;
		typename std::vector<T>::iterator		it_res;

		for (it_a=a.begin(), it_b=b.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_b++, it_res++)
			*it_res = *it_a + *it_b;

		return res;
	}

	/** A template function for substracting std::vector's having the same number of elements.
		*/
	template <class T>
	std::vector<T>  operator - (const std::vector<T> &a, const std::vector<T> &b)
	{
		ASSERT_(a.size()==b.size());

		typename std::vector<T>	res(a.size());
		typename std::vector<T>::const_iterator	it_a,it_b;
		typename std::vector<T>::iterator		it_res;

		for (it_a=a.begin(), it_b=b.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_b++, it_res++)
			*it_res = *it_a - *it_b;

		return res;
	}

	/** A template function for scalar product (element by element, like .* in MATLAB) of std::vector's having the same number of elements.
		*/
	template <class T>
	std::vector<T>  operator * (const std::vector<T> &a, const std::vector<T> &b)
	{
		ASSERT_(a.size()==b.size());

		typename std::vector<T>	res(a.size());
		typename std::vector<T>::const_iterator	it_a,it_b;
		typename std::vector<T>::iterator		it_res;

		for (it_a=a.begin(), it_b=b.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_b++, it_res++)
			*it_res = (*it_a) * (*it_b);

		return res;
	}

	/** A template function for scalar division (element by element, like ./ in MATLAB) of std::vector's having the same number of elements.
		*/
	template <class T>
	std::vector<T>  operator / (const std::vector<T> &a, const std::vector<T> &b)
	{
		ASSERT_(a.size()==b.size());

		typename std::vector<T>	res(a.size());
		typename std::vector<T>::const_iterator	it_a,it_b;
		typename std::vector<T>::iterator		it_res;

		for (it_a=a.begin(), it_b=b.begin(), it_res=res.begin(); it_a!=a.end(); it_a++, it_b++, it_res++)
			*it_res = (*it_a) / (*it_b);

		return res;
	}

	/** A template for the operator += applied to std::vector's having the same number of elements.
		*/
	template <class T>
	void  operator += (std::vector<T> &a, const std::vector<T> &b)
	{
		ASSERT_(a.size()==b.size());

		typename std::vector<T>::iterator			it_a;
		typename std::vector<T>::const_iterator		it_b;

		for (it_a=a.begin(), it_b=b.begin(); it_a!=a.end(); it_a++, it_b++)
			*it_a += *it_b;
	}

	/** A template for the operator -= applied to std::vector's having the same number of elements.
		*/
	template <class T>
	void  operator -= (std::vector<T> &a, const std::vector<T> &b)
	{
		ASSERT_(a.size()==b.size());

		typename std::vector<T>::iterator			it_a;
		typename std::vector<T>::const_iterator		it_b;

		for (it_a=a.begin(), it_b=b.begin(); it_a!=a.end(); it_a++, it_b++)
			*it_a -= *it_b;
	}

	/** A template for the operator *= applied to std::vector's having the same number of elements.
		*/
	template <class T>
	void  operator *= (std::vector<T> &a, const std::vector<T> &b)
	{
		ASSERT_(a.size()==b.size());

		typename std::vector<T>::iterator			it_a;
		typename std::vector<T>::const_iterator		it_b;

		for (it_a=a.begin(), it_b=b.begin(); it_a!=a.end(); it_a++, it_b++)
			*it_a *= *it_b;
	}

	/** A template for the operator /= applied to std::vector's having the same number of elements.
		*/
	template <class T>
	void  operator /= (std::vector<T> &a, const std::vector<T> &b)
	{
		ASSERT_(a.size()==b.size());

		typename std::vector<T>::iterator			it_a;
		typename std::vector<T>::const_iterator		it_b;

		for (it_a=a.begin(), it_b=b.begin(); it_a!=a.end(); it_a++, it_b++)
		{
			if (*it_b==0)
				THROW_EXCEPTION("Division by zero: execution aborted!")
			*it_a /= *it_b;
		}
	}

	/** A template for the operator += applied to a std::vector and a scalar.
		*/
	template <class T>
	void  operator += (std::vector<T> &a, T b)
	{
		for (typename std::vector<T>::iterator	it=a.begin();it!=a.end();it++)  *it += b;
	}

	/** A template for the operator -= applied to a std::vector and a scalar.
		*/
	template <class T>
	void  operator -= (std::vector<T> &a, T b)
	{
		for (typename std::vector<T>::iterator	it=a.begin();it!=a.end();it++)  *it -= b;
	}

	/** A template for the operator *= applied to a std::vector and a scalar.
		*/
	template <class T>
	void  operator *= (std::vector<T> &a, T b)
	{
		for (typename std::vector<T>::iterator	it=a.begin();it!=a.end();it++)  *it *= b;
	}

	/** A template for the operator /= applied to a std::vector and a scalar.
		*/
	template <class T>
	void  operator /= (std::vector<T> &a, T b)
	{
		if (b==0)	THROW_EXCEPTION("Division by zero: execution aborted!")
		for (typename std::vector<T>::iterator	it=a.begin();it!=a.end();it++)  *it /= b;
	}

	/** A template for counting how many elements in an array are non-Zero.
		*/
	template <class T>
	size_t  countNonZero(const std::vector<T> &a)
	{
		typename std::vector<T>::const_iterator	it_a;
		size_t		count=0;
		for (it_a=a.begin(); it_a!=a.end(); it_a++) if (*it_a) count++;
		return count;
	}

} // end of namespace



#endif

