/*---------------------------------------------------------------
	FILE: CDisplayWindow.h
	USE: See description below.

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef  CDisplayWindow_H
#define  CDisplayWindow_H

#include <MRPT/UTILS/utils_defs.h>
#include <MRPT/config.h>
#include <MRPT/UTILS/SYNCH.h>

/*---------------------------------------------------------------
	Class
  ---------------------------------------------------------------*/
namespace UTILS
{
	class CMRPTImage;
	class CMRPTImageFloat;

	/** This class creates a window as a graphical user interface (GUI) for displaying images to the user.
	 *  From MRPT 0.4 (Sep 2007), this class is a wrapper for the cross-platform OpenCV's mechanism of named windows.
	 */
	class CDisplayWindow
	{
	protected:
		/** The caption of the window:
		  */
		std::string		m_caption;

		/** The OpenCV window "name"
		  */
        std::string     m_winName;

		/** Auxiliars
		  */
		volatile static bool	m_keyPushed;

		/** For the unique OpenCV names.
		  */
		static int      m_counter;
		static SYNCH::CCriticalSection  m_counterCS;

		/** For the GUI processing thread.
		  */
		static int      m_openWindows;
		static SYNCH::CCriticalSection  m_openWindowsCS;

		/** The thread that runs whenever 1+ windows are open:
		  */
		static void     threadGUI(void *);
		static unsigned long  m_threadGUI_id;

        void           updateWindowTitle();

	public:
		/** Constructor. This is the only point where a text title can be associated to the window.
		 */
		CDisplayWindow( const std::string &windowCaption = std::string() );

		/** Destructor
		 */
		~CDisplayWindow();

		/** Show a given color or grayscale image on the window.
		 *  It adapts the size of the window to that of the image if "winWidth" OR "winHeight" are "-1". Otherwise, the image is stretched to the given size.
		 */
		void  showImage(
			const CMRPTImage	&img,
			int					winWidth = -1,
			int					winHeight = -1
			);

		/** Show a given grayscale image on the window.
		 *  It adapts the size of the window to that of the image if "winWidth" OR "winHeight" are "-1". Otherwise, the image is stretched to the given size.
		 */
		void  showImage(
			const CMRPTImageFloat	&img,
			int						winWidth = -1,
			int						winHeight = -1
			);

		/** Resizes the window, stretching the image to fit into the display area.
		 */
		void  resize( unsigned int width, unsigned int height );

		/** Changes the position of the window on the screen.
		 */
		void  setPos( int x, int y );

		/** Waits for any key to be pushed on the image
		  */
		void  waitForKey( );


	}; // End of class def.

} // End of namespace

#endif
