/*---------------------------------------------------------------
	FILE: CPosePDFGrid.h
	USE: See above.

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef CPosePDFGrid_H
#define CPosePDFGrid_H

#include <MRPT/MRML/CPosePDF.h>
#include <MRPT/MRML/CPose2DGridTemplate.h>

namespace MRML
{

	/** Declares a class that represents a Probability Distribution
	 *    function (PDF) of a 2D pose (x,y,phi).
	 *   This class implements that PDF using a 3D grid.
     *
	 * \sa CPose2D, CPosePDF, CPose2DGridTemplate
	 */
	class CPosePDFGrid : public CPosePDF, public CPose2DGridTemplate<float>
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CPosePDFGrid )

	 protected:


	 public:
		/** Constructor: Initializes a, uniform distribution over the whole given range.
		  */
		CPosePDFGrid(
			float		xMin = -1.0f,
			float		xMax = 1.0f,
			float		yMin = -1.0f,
			float		yMax = 1.0f,
			float		resolutionXY = 0.5f,
			float		resolutionPhi = DEG2RAD(180),
			float		phiMin = -M_PIf,
			float		phiMax = M_PIf
			);

		/** Destructor
		 */
		virtual ~CPosePDFGrid();

		/** Copy operator, translating if necesary (for example, between particles and gaussian representations)
		  */
		void  copyFrom(const CPosePDF &o);

		/** Normalizes the PDF, such as all cells sum the unity.
		  */
		void  normalize();

		/** Assigns the same value to all the cells in the grid, so the sum 1.
		  */
		void  uniformDistribution();

		 /** Returns an estimate of the pose, (the mean, or mathematical expectation of the PDF), computed
		  *   as a weighted average over all particles.
		  */
		CPose2D  getEstimatedPose() const;

		/** Returns an estimate of the pose covariance matrix (3x3 cov.matrix  for x,y,phi variables)
		  */
		CMatrix  getEstimatedCovariance() const;

		/** Save the contents of the 3D grid in one file, as a vertical concatenation of rectangular matrix for the different "PHI" discrete levels, and the size in X,Y,and PHI in another file named "<filename>_dims.txt"
		 */
		void   saveToTextFile(char *dataFile) const;

		/** This can be used to convert a PDF from local coordinates to global, providing the point (newReferenceBase) from which
		  *   "to project" the current pdf. Result PDF substituted the currently stored one in the object.
		  */
		void  changeCoordinatesReference( const CPose2D &newReferenceBase );

		/** Bayesian fusion of 2 densities (In the grid representation this becomes a pointwise multiplication)
		  */
		void  bayesianFusion( CPosePDF &p1, CPosePDF &p2 );

		/** Returns a new PDF such as: NEW_PDF = (0,0,0) - THIS_PDF
		  */
		void  inverse(CPosePDF &o) const;

		/** Draws a single sample from the distribution (WARNING: weights are assumed to be normalized!)
		  */
		void  drawSingleSample( CPose2D &outPart ) const;

		/** Draws a number of samples from the distribution, and saves as a list of 1x3 vectors, where each row contains a (x,y,phi) datum.
		  */
		void  drawManySamples( size_t N, std::vector<vector_float> & outSamples ) const;

	}; // End of class def.

} // End of namespace

#endif
