/*---------------------------------------------------------------
	FILE: CPosePDFGaussian.h
	USE: See above.

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef CPosePDFGaussian_H
#define CPosePDFGaussian_H

#include <MRPT/MRML/CPosePDF.h>
#include <MRPT/UTILS/CMatrix.h>

namespace MRML
{
	/** Declares a class that represents a Probability Density  function (PDF) of a 2D pose \f$ p(\mathbf{x}) = [x ~ y ~ \phi ]^t \f$.
	 *
	 *   This class implements that PDF using a mono-modal Gaussian distribution. See MRML::CPosePDF for more details.
     *
	 * \sa CPose2D, CPosePDF, CPosePDFParticles
	 */
	class CPosePDFGaussian : public CPosePDF
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CPosePDFGaussian )

	protected:
		/** Assures the symmetry of the covariance matrix (eventually certain operations in the math-coprocessor lead to non-symmetric matrixes!)
		  */
		void  assureSymmetry();

	 public:
		/** Default constructor
		  */
		CPosePDFGaussian();

		/** Constructor
		  */
		CPosePDFGaussian( const CPose2D &init_Mean );

		/** Constructor
		  */
		CPosePDFGaussian( const CPose2D &init_Mean, const CMatrix &init_Cov );

		/** The mean value
		 */
		CPose2D		mean;

		/** The 3x3 covariance matrix
		 */
		CMatrix	cov;


		 /** Returns an estimate of the pose, (the mean, or mathematical expectation of the PDF)
		  */
		CPose2D  getEstimatedPose() const;

		 /** Returns an estimate of the pose covariance matrix (3x3 cov.matrix  for x,y,phi variables)
		  */
		CMatrix  getEstimatedCovariance() const;

		/** Copy operator, translating if necesary (for example, between particles and gaussian representations)
		  */
		void  copyFrom(const CPosePDF &o);

		/** Save PDF's particles to a text file, containing the 2D pose in the first line, then the covariance matrix in next 3 lines.
		 */
		void  saveToTextFile(char *file) const;

		/** This can be used to convert a PDF from local coordinates to global, providing the point (newReferenceBase) from which
		  *   "to project" the current pdf. Result PDF substituted the currently stored one in the object.
		  */
		void  changeCoordinatesReference( const CPose2D &newReferenceBase );

		/** Rotate the covariance matrix by replacing it by \f$ \mathbf{R}~\mathbf{COV}~\mathbf{R}^t \f$, where \f$ \mathbf{R} = \left[ \begin{array}{ccc} \cos\alpha & -\sin\alpha & 0 \\ \sin\alpha & \cos\alpha & 0 \\ 0 & 0 & 1 \end{array}\right] \f$.
		  */
		void  rotateCov(float ang);

		/** Draws a single sample from the distribution
		  */
		void  drawSingleSample( CPose2D &outPart ) const;

		/** Draws a number of samples from the distribution, and saves as a list of 1x3 vectors, where each row contains a (x,y,phi) datum.
		  */
		void  drawManySamples( size_t N, std::vector<vector_float> & outSamples ) const;

		/** Bayesian fusion of two points gauss. distributions, then save the result in this object.
		  *  The process is as follows:<br>
		  *		- (x1,S1): Mean and variance of the p1 distribution.
		  *		- (x2,S2): Mean and variance of the p2 distribution.
		  *		- (x,S): Mean and variance of the resulting distribution.
		  *
		  *    S = (S1<sup>-1</sup> + S2<sup>-1</sup>)<sup>-1</sup>;
		  *    x = S * ( S1<sup>-1</sup>*x1 + S2<sup>-1</sup>*x2 );
		  */
		void  bayesianFusion( CPosePDF &p1, CPosePDF &p2 );

		/** Returns a new PDF such as: NEW_PDF = (0,0,0) - THIS_PDF
		  */
		void	 inverse(CPosePDF &o) const;

		/** Makes: thisPDF = thisPDF + Ap, where "+" is pose composition (both the mean, and the covariance matrix are updated).
		  */
		void  operator += ( CPose2D Ap);

		/** Evaluates the PDF at a given point.
		  */
		double  evaluatePDF( const CPose2D &x ) const;

		/** Evaluates the ratio PDF(x) / PDF(MEAN), that is, the normalized PDF in the range [0,1].
		  */
		double  evaluateNormalizedPDF( const CPose2D &x ) const;

		/** Computes the Mahalanobis distance between the centers of two Gaussians.
		  */
		double  mahalanobisDistanceTo( const CPosePDFGaussian& theOther );

		/** Substitutes the diagonal elements if (square) they are below some given minimum values (Use this before bayesianFusion, for example, to avoid inversion of singular matrixes, etc...)
		  */
		void  assureMinCovariance( float minStdXY, float minStdPhi );

	}; // End of class def.

	/** Dumps the mean and covariance matrix to a text stream.
	  */
	std::ostream &  operator << (std::ostream & out, const CPosePDFGaussian& obj);

	/** Returns the Gaussian distribution of \f$ \mathbf{C} \f$, for \f$ \mathbf{C} = \mathbf{A} \oplus \mathbf{B} \f$.
	  */
	MRML::CPosePDFGaussian	 operator + ( const MRML::CPose2D &A, const MRML::CPosePDFGaussian &B  );

} // End of namespace

#endif
