// Malloc_T.cpp
// Malloc_T.cpp,v 4.66 2000/02/10 19:57:38 nanbor Exp

#ifndef ACE_MALLOC_T_C
#define ACE_MALLOC_T_C

#define ACE_BUILD_DLL
#include "ace/Malloc_T.h"

#if !defined (ACE_LACKS_PRAGMA_ONCE)
# pragma once
#endif /* ACE_LACKS_PRAGMA_ONCE */

#if !defined (__ACE_INLINE__)
#include "ace/Malloc_T.i"
#endif /* __ACE_INLINE__ */

ACE_RCSID(ace, Malloc_T, "Malloc_T.cpp,v 4.66 2000/02/10 19:57:38 nanbor Exp")

template <class T, class ACE_LOCK>
ACE_Cached_Allocator<T, ACE_LOCK>::ACE_Cached_Allocator (size_t n_chunks)
  : pool_ (0),
    free_list_ (ACE_PURE_FREE_LIST)
{
  ACE_NEW (this->pool_,
           char[n_chunks * sizeof (T)]);

  for (size_t c = 0;
       c < n_chunks;
       c++)
    {
      void* placement = this->pool_ + c * sizeof(T);
      this->free_list_.add (new (placement) ACE_Cached_Mem_Pool_Node<T>);
    }
  // Put into free list using placement contructor, no real memory
  // allocation in the above <new>.
}

template <class T, class ACE_LOCK>
ACE_Cached_Allocator<T, ACE_LOCK>::~ACE_Cached_Allocator (void)
{
  delete [] this->pool_;
}

ACE_ALLOC_HOOK_DEFINE (ACE_Malloc_T)

template <class MALLOC>
ACE_Allocator_Adapter<MALLOC>::ACE_Allocator_Adapter (LPCTSTR pool_name)
  : allocator_ (pool_name)
{
  ACE_TRACE ("ACE_Allocator_Adapter<MALLOC>::ACE_Allocator_Adapter");
}

template <class MALLOC>
ACE_Allocator_Adapter<MALLOC>::~ACE_Allocator_Adapter (void)
{
  ACE_TRACE ("ACE_Allocator_Adapter<MALLOC>::~ACE_Allocator_Adapter");
}

#if defined (ACE_HAS_MALLOC_STATS)
template <class MALLOC> void
ACE_Allocator_Adapter<MALLOC>::print_stats (void) const
{
  ACE_TRACE ("ACE_Allocator_Adaptor<MALLOC>::print_stats");
  this->allocator_.print_stats ();
}
#endif /* ACE_HAS_MALLOC_STATS */

template <class MALLOC> void
ACE_Allocator_Adapter<MALLOC>::dump (void) const
{
  ACE_TRACE ("ACE_Malloc<MALLOC>::dump");
  this->allocator_.dump ();
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> void
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::dump (void) const
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::dump");

  ACE_DEBUG ((LM_DEBUG, ACE_BEGIN_DUMP, this));
  this->memory_pool_.dump ();
  ACE_DEBUG ((LM_DEBUG, ASYS_TEXT ("cb_ptr_ = %x"), this->cb_ptr_));
  ACE_DEBUG ((LM_DEBUG, ASYS_TEXT ("\n")));
  this->cb_ptr_->dump ();
#if defined (ACE_HAS_MALLOC_STATS)
  if (this->cb_ptr_ != 0)
    this->cb_ptr_->malloc_stats_.dump ();
#endif /* ACE_HAS_MALLOC_STATS */
  ACE_DEBUG ((LM_DEBUG, ACE_END_DUMP));
}

#if defined (ACE_HAS_MALLOC_STATS)

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> void
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::print_stats (void) const
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::print_stats");
  ACE_GUARD (ACE_LOCK, ace_mon, (ACE_LOCK &) this->lock_);

  if (this->cb_ptr_ == 0)
    return;
  this->cb_ptr_->malloc_stats_.dump ();
  ACE_DEBUG ((LM_DEBUG,
              ASYS_TEXT ("(%P|%t) contents of freelist:\n")));

  for (MALLOC_HEADER *currp = this->cb_ptr_->freep_->next_block_;
       ;
       currp = currp->next_block_)
    {
      ACE_DEBUG ((LM_DEBUG,
                  ASYS_TEXT ("(%P|%t) ptr = %u, MALLOC_HEADER units = %d, byte units = %d\n"),
                  currp,
                  currp->size_,
                  currp->size_ * sizeof (MALLOC_HEADER)));
      if (currp == this->cb_ptr_->freep_)
        break;
    }
}
#endif /* ACE_HAS_MALLOC_STATS */

// Put <ptr> in the free list (locked version).

template<ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> void
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::free (void *ptr)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::free");
  ACE_GUARD (ACE_LOCK, ace_mon, this->lock_);

  this->shared_free (ptr);
}

// This function is called by the ACE_Malloc_T constructor to initialize
// the memory pool.  The first time in it allocates room for the
// control block (as well as a chunk of memory, depending on
// rounding...).  Depending on the type of <MEM_POOL> (i.e., shared
// vs. local) subsequent calls from other processes will only
// initialize the control block pointer.

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::open (void)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::open");
  ACE_GUARD_RETURN (ACE_LOCK, ace_mon, this->lock_, -1);

  size_t rounded_bytes = 0;
  int first_time = 0;

  this->cb_ptr_ = (ACE_CB *)
    this->memory_pool_.init_acquire (sizeof *this->cb_ptr_,
                                     rounded_bytes,
                                     first_time);
  if (this->cb_ptr_ == 0)
    ACE_ERROR_RETURN ((LM_ERROR,
                       ASYS_TEXT ("(%P|%t) %p\n"),
                       ASYS_TEXT ("init_acquire failed")),
                      -1);
  else if (first_time)
    {
      // ACE_DEBUG ((LM_DEBUG, ASYS_TEXT ("(%P|%t) first time in, control block = %u\n"), this->cb_ptr_));

#if defined (ACE_HAS_MALLOC_STATS)
      // Call the constructor on the ACE_LOCK, using the placement
      // operator!
      new ((void *) &this->cb_ptr_->malloc_stats_) ACE_Malloc_Stats;
#endif /* ACE_HAS_MALLOC_STATS */

      MALLOC_HEADER::init_ptr (&this->cb_ptr_->freep_,
                               &this->cb_ptr_->base_,
                               this->cb_ptr_);

      MALLOC_HEADER::init_ptr (&this->cb_ptr_->freep_->next_block_,
                               this->cb_ptr_->freep_,
                               this->cb_ptr_);

      NAME_NODE::init_ptr (&this->cb_ptr_->name_head_,
                           0,
                           this->cb_ptr_);

      this->cb_ptr_->freep_->size_ = 0;

      if (rounded_bytes > (sizeof *this->cb_ptr_ + sizeof (MALLOC_HEADER)))
        {
          // If we've got any extra space at the end of the control
          // block, then skip past the dummy <MALLOC_HEADER> to
          // point at the first free block.
          MALLOC_HEADER *p = ((MALLOC_HEADER *) (this->cb_ptr_->freep_)) + 1;

          MALLOC_HEADER::init_ptr (&p->next_block_,
                                   0,
                                   this->cb_ptr_);

          // Why aC++ in 64-bit mode can't grok this, I have no
          // idea... but it ends up with an extra bit set which makes
          // size_ really big without this hack.
#if defined (__hpux) && defined (__LP64__)
          size_t hpux11_hack = (rounded_bytes - sizeof *this->cb_ptr_)
                               / sizeof (MALLOC_HEADER);
          p->size_ = hpux11_hack;
#else
          p->size_ = (rounded_bytes - sizeof *this->cb_ptr_)
            / sizeof (MALLOC_HEADER);
#endif /* (__hpux) && defined (__LP64__) */

          ACE_MALLOC_STATS (++this->cb_ptr_->malloc_stats_.nchunks_);
          ACE_MALLOC_STATS (++this->cb_ptr_->malloc_stats_.nblocks_);
          ACE_MALLOC_STATS (++this->cb_ptr_->malloc_stats_.ninuse_);

          // Insert the newly allocated chunk of memory into the free
          // list.  Add "1" to skip over the <MALLOC_HEADER> when
          // freeing the pointer.
          this->shared_free (p + 1);
        }
    }
  return 0;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB>
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_T (LPCTSTR pool_name)
  : memory_pool_ (pool_name),
    lock_ (pool_name == 0 ? 0 : ACE::basename (pool_name,
                                               ACE_DIRECTORY_SEPARATOR_CHAR))
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_T");
  if (this->open () == -1)
    ACE_ERROR ((LM_ERROR,
                ASYS_TEXT ("%p\n"),
                ASYS_TEXT ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_T")));
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB>
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_T (LPCTSTR pool_name,
                                                          LPCTSTR lock_name,
                                                          const ACE_MEM_POOL_OPTIONS *options)
  : memory_pool_ (pool_name, options),
    lock_ (lock_name != 0 ? lock_name : ACE::basename (pool_name,
                                                       ACE_DIRECTORY_SEPARATOR_CHAR))
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_T");
  if (this->open () == -1)
    ACE_ERROR ((LM_ERROR,
                ASYS_TEXT ("%p"),
                ASYS_TEXT ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_T")));
}

#if !defined (ACE_HAS_TEMPLATE_TYPEDEFS)
template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB>
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_T (LPCTSTR pool_name,
                                                          LPCTSTR lock_name,
                                                          const void *options)
  : memory_pool_ (pool_name,
                  (const ACE_MEM_POOL_OPTIONS *) options),
    lock_ (lock_name)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_T");
  if (this->open () == -1)
    ACE_ERROR ((LM_ERROR,
                ASYS_TEXT ("%p"),
                ASYS_TEXT ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_T")));
}
#endif /* ACE_HAS_TEMPLATE_TYPEDEFS */


template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB>
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::~ACE_Malloc_T (void)
{
  ACE_TRACE ("ACE_Malloc_T<MEM_POOL>::~ACE_Malloc_T<MEM_POOL>");
}

// Clean up the resources allocated by ACE_Malloc_T.

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::remove (void)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::remove");
  // ACE_DEBUG ((LM_DEBUG, ASYS_TEXT ("(%P|%t) destroying ACE_Malloc_T\n")));
  int result = 0;

#if defined (ACE_HAS_MALLOC_STATS)
  this->print_stats ();

  ACE_DES_NOFREE (this->cb_ptr_->malloc_stats_,
                  ACE_Malloc_Stats);
#endif /* ACE_HAS_MALLOC_STATS */

  // Remove the ACE_LOCK.
  this->lock_.remove ();

  // Give the memory pool a chance to release its resources.
  result = this->memory_pool_.release ();

  return result;
}

// General-purpose memory allocator.  Assumes caller holds the locks.

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> void *
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::shared_malloc (size_t nbytes)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::shared_malloc");

  if (this->cb_ptr_ == 0)
    return 0;

  // Round up request to a multiple of the MALLOC_HEADER size.
  size_t nunits =
    (nbytes + sizeof (MALLOC_HEADER) - 1) / sizeof (MALLOC_HEADER)
    + 1; // Add one for the <MALLOC_HEADER> itself.

  MALLOC_HEADER *prevp = 0;
  MALLOC_HEADER *currp = 0;

  ACE_SEH_TRY
    {
      // Begin the search starting at the place in the freelist where the
      // last block was found.
      prevp = this->cb_ptr_->freep_;
      currp = prevp->next_block_;
    }
  ACE_SEH_EXCEPT (this->memory_pool_.seh_selector (GetExceptionInformation ()))
    {
    }

  // Search the freelist to locate a block of the appropriate size.


  while (1)

    // *Warning* Do not use "continue" within this while-loop.

    {
      ACE_SEH_TRY
        {
          if (currp->size_ >= nunits) // Big enough
            {
              ACE_MALLOC_STATS (++this->cb_ptr_->malloc_stats_.ninuse_);
              if (currp->size_ == nunits)
                // Exact size, just update the pointers.
                prevp->next_block_ = currp->next_block_;
              else
                {
                  // Remaining chunk is larger than requested block, so
                  // allocate at tail end.
                  ACE_MALLOC_STATS (++this->cb_ptr_->malloc_stats_.nblocks_);
                  currp->size_ -= nunits;
                  currp += currp->size_;
                  MALLOC_HEADER::init_ptr (&currp->next_block_,
                                           0,
                                           this->cb_ptr_);
                  currp->size_ = nunits;
                }
              this->cb_ptr_->freep_ = prevp;

              // Skip over the MALLOC_HEADER when returning pointer.
              return currp + 1;
            }
          else if (currp == this->cb_ptr_->freep_)
            {
              // We've wrapped around freelist without finding a
              // block.  Therefore, we need to ask the memory pool for
              // a new chunk of bytes.

              size_t chunk_bytes = 0;

              currp = (MALLOC_HEADER *)
                this->memory_pool_.acquire (nunits * sizeof (MALLOC_HEADER),
                                            chunk_bytes);
              if (currp != 0)
                {
                  ACE_MALLOC_STATS (++this->cb_ptr_->malloc_stats_.nblocks_);
                  ACE_MALLOC_STATS (++this->cb_ptr_->malloc_stats_.nchunks_);
                  ACE_MALLOC_STATS (++this->cb_ptr_->malloc_stats_.ninuse_);

                  MALLOC_HEADER::init_ptr (&currp->next_block_,
                                           0,
                                           this->cb_ptr_);
                  // Compute the chunk size in MALLOC_HEADER units.
                  currp->size_ = chunk_bytes / sizeof (MALLOC_HEADER);

                  // Insert the newly allocated chunk of memory into the
                  // free list.  Add "1" to skip over the
                  // <MALLOC_HEADER> when freeing the pointer since
                  // the first thing <free> does is decrement by this
                  // amount.
                  this->shared_free (currp + 1);
                  currp = this->cb_ptr_->freep_;
                }
              else
                ACE_ERROR_RETURN ((LM_ERROR,
                                   ASYS_TEXT ("(%P|%t) %p\n"),
                                   ASYS_TEXT ("malloc")),
                                  0);
            }
          prevp = currp;
          currp = currp->next_block_;
        }
      ACE_SEH_EXCEPT (this->memory_pool_.seh_selector (GetExceptionInformation ()))
        {
        }
    }
  ACE_NOTREACHED (return 0;)
}

// General-purpose memory allocator.

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> void *
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::malloc (size_t nbytes)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::malloc");
  ACE_GUARD_RETURN (ACE_LOCK, ace_mon, this->lock_, 0);

  return this->shared_malloc (nbytes);
}

// General-purpose memory allocator.

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> void *
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::calloc (size_t nbytes,
                                                      char initial_value)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::calloc");
  void *ptr = this->malloc (nbytes);

  if (ptr != 0)
    ACE_OS::memset (ptr, initial_value, nbytes);

  return ptr;
}

// Put block AP in the free list (must be called with locks held!)

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> void
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::shared_free (void *ap)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::shared_free");

  if (ap == 0 || this->cb_ptr_ == 0)
    return;


  // Adjust AP to point to the block MALLOC_HEADER
  MALLOC_HEADER *blockp = ((MALLOC_HEADER *) ap) - 1;
  MALLOC_HEADER *currp = this->cb_ptr_->freep_;

  // Search until we find the location where the blocks belongs.  Note
  // that addresses are kept in sorted order.

  ACE_SEH_TRY
    {
      for (;
           blockp <= currp
             || blockp >= (MALLOC_HEADER *) currp->next_block_;
           currp = currp->next_block_)
        {
          if (currp >= (MALLOC_HEADER *) currp->next_block_
              && (blockp > currp
                  || blockp < (MALLOC_HEADER *) currp->next_block_))
            // Freed block at the start or the end of the memory pool.
            break;
        }

      // Join to upper neighbor.
      if ((blockp + blockp->size_) == currp->next_block_)
        {
          ACE_MALLOC_STATS (--this->cb_ptr_->malloc_stats_.nblocks_);
          blockp->size_ += currp->next_block_->size_;
          blockp->next_block_ = currp->next_block_->next_block_;
        }
      else
        blockp->next_block_ = currp->next_block_;

      // Join to lower neighbor.
      if ((currp + currp->size_) == blockp)
        {
          ACE_MALLOC_STATS (--this->cb_ptr_->malloc_stats_.nblocks_);
          currp->size_ += blockp->size_;
          currp->next_block_ = blockp->next_block_;
        }
      else
        currp->next_block_ = blockp;

      ACE_MALLOC_STATS (--this->cb_ptr_->malloc_stats_.ninuse_);
      this->cb_ptr_->freep_ = currp;
    }
  ACE_SEH_EXCEPT (this->memory_pool_.seh_selector (GetExceptionInformation ()))
    {
    }
}

// No locks held here, caller must acquire/release lock.

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> void*
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::shared_find (const char *name)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::shared_find");

  if (this->cb_ptr_ == 0)
    return 0;

  ACE_SEH_TRY
    {
      for (NAME_NODE *node = this->cb_ptr_->name_head_;
           node != 0;
           node = node->next_)
        if (ACE_OS::strcmp (node->name (),
                            name) == 0)
          return node;
    }
  ACE_SEH_EXCEPT (this->memory_pool_.seh_selector (GetExceptionInformation ()))
    {
    }
  return 0;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::shared_bind (const char *name,
                                                             void *pointer)
{
  if (this->cb_ptr_ == 0)
    return -1;

  // Combine the two allocations into one to avoid overhead...
  NAME_NODE *new_node = 0;

  ACE_ALLOCATOR_RETURN (new_node,
                        (NAME_NODE *)
                        this->shared_malloc (sizeof (NAME_NODE) +
                                             ACE_OS::strlen (name) + 1),
                        -1);
  char *name_ptr = (char *) (new_node + 1);

  // Use operator placement new to insert <new_node> at the head of
  // the linked list of <NAME_NODE>s.
  NAME_NODE *result =
    new (new_node) NAME_NODE (name,
                              name_ptr,
                              ACE_reinterpret_cast (char *,
                                                    pointer),
                              this->cb_ptr_->name_head_);
  this->cb_ptr_->name_head_ = result;
  return 0;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::trybind (const char *name,
                                                         void *&pointer)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::trybind");
  ACE_WRITE_GUARD_RETURN (ACE_LOCK, ace_mon, this->lock_, -1);

  NAME_NODE *node = (NAME_NODE *) this->shared_find (name);

  if (node == 0)
    // Didn't find it, so insert it.
    return this->shared_bind (name, pointer);
  else
    {
      // Found it, so return a copy of the current entry.
      pointer = (char *) node->pointer_;
      return 1;
    }
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::bind (const char *name,
                                                      void *pointer,
                                                      int duplicates)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::bind");
  ACE_WRITE_GUARD_RETURN (ACE_LOCK, ace_mon, this->lock_, -1);

  if (duplicates == 0 && this->shared_find (name) != 0)
    // If we're not allowing duplicates, then if the name is already
    // present, return 1.
    return 1;
  else
    // If we get this far, either we're allowing duplicates or we didn't
    // find the name yet.

    return this->shared_bind (name, pointer);
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::find (const char *name,
                                                    void *&pointer)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::find");

  ACE_READ_GUARD_RETURN (ACE_LOCK, ace_mon, this->lock_, -1);

  NAME_NODE *node = (NAME_NODE *) this->shared_find (name);

  if (node == 0)
    return -1;
  else
    {
      pointer = (char *) node->pointer_;
      return 0;
    }
}

// Returns a count of the number of available chunks that can hold
// <size> byte allocations.  Function can be used to determine if you
// have reached a water mark. This implies a fixed amount of allocated
// memory.
//
// @param size - the chunk size of that you would like a count of
// @return function returns the number of chunks of the given size
//          that would fit in the currently allocated memory

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> ssize_t
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::avail_chunks (size_t size) const
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::avail_chunks");
  ACE_READ_GUARD_RETURN (ACE_LOCK, ace_mon, (ACE_LOCK &) this->lock_, -1);

  if (this->cb_ptr_ == 0)
    return -1;

  size_t count = 0;
  // Avoid dividing by 0...
  size = size == 0 ? 1 : size;

  for (MALLOC_HEADER *currp = this->cb_ptr_->freep_->next_block_;
       currp != this->cb_ptr_->freep_;
       currp = currp->next_block_)
    // Calculate how many will fit in this block.
    if (currp->size_ * sizeof (MALLOC_HEADER) >= size)
      count += currp->size_ * sizeof (MALLOC_HEADER) / size;

  return count;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::find (const char *name)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::find");
  ACE_READ_GUARD_RETURN (ACE_LOCK, ace_mon, this->lock_, -1);

  return this->shared_find (name) == 0 ? -1 : 0;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::unbind (const char *name, void *&pointer)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::unbind");
  ACE_WRITE_GUARD_RETURN (ACE_LOCK, ace_mon, this->lock_, -1);

  if (this->cb_ptr_ == 0)
    return -1;

  NAME_NODE *prev = 0;

  for (NAME_NODE *curr = this->cb_ptr_->name_head_;
       curr != 0;
       curr = curr->next_)
    {
      if (ACE_OS::strcmp (curr->name (), name) == 0)
        {
          pointer = (char *) curr->pointer_;

          if (prev == 0)
            this->cb_ptr_->name_head_ = curr->next_;
          else
            prev->next_ = curr->next_;

          if (curr->next_)
            curr->next_->prev_ = prev;

          // This will free up both the node and the name due to our
          // clever trick in <bind>!
          this->shared_free (curr);
          return 0;
        }
      prev = curr;
    }

  // Didn't find it, so fail.
  return -1;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::unbind (const char *name)
{
  ACE_TRACE ("ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::unbind");
  void *temp = 0;
  return this->unbind (name, temp);
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> void
ACE_Malloc_LIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::dump (void) const
{
  ACE_TRACE ("ACE_Malloc_LIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::dump");

  ACE_DEBUG ((LM_DEBUG, ACE_BEGIN_DUMP, this));
  this->curr_->dump ();
  this->guard_.dump ();
  ACE_DEBUG ((LM_DEBUG, ASYS_TEXT ("name_ = %s"), this->name_));
  ACE_DEBUG ((LM_DEBUG, ASYS_TEXT ("\n")));
  ACE_DEBUG ((LM_DEBUG, ACE_END_DUMP));
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB>
ACE_Malloc_LIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_LIFO_Iterator_T (ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB> &malloc,
                                                                                      const char *name)
  : malloc_ (malloc),
    curr_ (0),
    guard_ (malloc_.lock_),
    name_ (name != 0 ? ACE_OS::strdup (name) : 0)
{
  ACE_TRACE ("ACE_Malloc_LIFO_Iterator<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_LIFO_Iterator");
  // Cheap trick to make code simple.
  // @@ Doug, this looks like trouble...
  NAME_NODE temp;
  this->curr_ = &temp;
  this->curr_->next_ = malloc_.cb_ptr_->name_head_;

  this->advance ();
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB>
ACE_Malloc_LIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::~ACE_Malloc_LIFO_Iterator_T (void)
{
  ACE_OS::free ((void *) this->name_);
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_LIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::next (void *&next_entry,
                                                                    const char *&name)
{
  ACE_TRACE ("ACE_Malloc_LIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::next");

  if (this->curr_ != 0)
    {
      next_entry = (char *) this->curr_->pointer_;
      name = this->curr_->name ();
      return 1;
    }
  else
    return 0;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_LIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::next (void *&next_entry)
{
  ACE_TRACE ("ACE_Malloc_LIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::next");

  if (this->curr_ != 0)
    {
      next_entry = this->curr_->pointer_;
      return 1;
    }
  else
    return 0;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_LIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::done (void) const
{
  ACE_TRACE ("ACE_Malloc_LIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::done");

  return this->curr_ == 0;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_LIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::advance (void)
{
  ACE_TRACE ("ACE_Malloc_LIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::advance");

  this->curr_ = this->curr_->next_;

  if (this->name_ == 0)
    return this->curr_ != 0;

  while (this->curr_ != 0
         && ACE_OS::strcmp (this->name_,
                            this->curr_->name ()) != 0)
    this->curr_ = this->curr_->next_;

  return this->curr_ != 0;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> void
ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::dump (void) const
{
  ACE_TRACE ("ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::dump");

  ACE_DEBUG ((LM_DEBUG, ACE_BEGIN_DUMP, this));
  this->curr_->dump ();
  this->guard_.dump ();
  ACE_DEBUG ((LM_DEBUG, ASYS_TEXT ("name_ = %s"), this->name_));
  ACE_DEBUG ((LM_DEBUG, ASYS_TEXT ("\n")));
  ACE_DEBUG ((LM_DEBUG, ACE_END_DUMP));
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB>
ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_FIFO_Iterator_T (ACE_Malloc_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB> &malloc,
                                                                                      const char *name)
  : malloc_ (malloc),
    curr_ (0),
    guard_ (malloc_.lock_),
    name_ (name != 0 ? ACE_OS::strdup (name) : 0)
{
  ACE_TRACE ("ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::ACE_Malloc_FIFO_Iterator");
  // Cheap trick to make code simple.
  // @@ Doug, this looks like trouble...
  NAME_NODE temp;
  this->curr_ = &temp;
  this->curr_->next_ = malloc_.cb_ptr_->name_head_;
  this->curr_->prev_ = 0;

  // Go to the first element that was inserted.
  this->start ();
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB>
ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::~ACE_Malloc_FIFO_Iterator_T (void)
{
  ACE_OS::free ((void *) this->name_);
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::next (void *&next_entry,
                                                                    const char *&name)
{
  ACE_TRACE ("ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::next");

  if (this->curr_ != 0)
    {
      next_entry = (char *) this->curr_->pointer_;
      name = this->curr_->name ();
      return 1;
    }
  else
    return 0;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::next (void *&next_entry)
{
  ACE_TRACE ("ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::next");

  if (this->curr_ != 0)
    {
      next_entry = this->curr_->pointer_;
      return 1;
    }
  else
    return 0;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::done (void) const
{
  ACE_TRACE ("ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::done");

  return this->curr_ == 0;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::advance (void)
{
  ACE_TRACE ("ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::advance");

  this->curr_ = this->curr_->prev_;

  if (this->name_ == 0)
    return this->curr_ != 0;

  while (this->curr_ != 0
         && ACE_OS::strcmp (this->name_,
                            this->curr_->name ()) != 0)
    this->curr_ = this->curr_->prev_;

  return this->curr_ != 0;
}

template <ACE_MEM_POOL_1, class ACE_LOCK, class ACE_CB> int
ACE_Malloc_FIFO_Iterator_T<ACE_MEM_POOL_2, ACE_LOCK, ACE_CB>::start (void)
{
  this->curr_ = this->curr_->next_;
  NAME_NODE *prev = 0;

  // Locate the element that was inserted first.
  // @@ We could optimize this by making the list a circular list or
  // storing an extra pointer.
  while (this->curr_ != 0)
    {
      prev = this->curr_;
      this->curr_ = this->curr_->next_;
    }

  this->curr_ = prev;
  return this->curr_ != 0;
}

#endif /* ACE_MALLOC_T_C */
