/*---------------------------------------------------------------
	FILE: CPose3DPDFSOG.h
	USE: See above.

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef CPose3DPDFSOG_H
#define CPose3DPDFSOG_H

#include <MRPT/MRML/CPose3DPDF.h>
#include <MRPT/MRML/CPose3DPDFGaussian.h>
#include <MRPT/UTILS/CMatrix.h>

namespace MRML
{
	/** Declares a class that represents a Probability Density function (PDF) of a 3D(6D) pose \f$ p(\mathbf{x}) = [x ~ y ~ z ~ yaw ~ pitch ~ roll]^t \f$.
	 *   This class implements that PDF as the following multi-modal Gaussian distribution:
	 *
	 * \f$ p(\mathbf{x}) = \sum\limits_{i=1}^N \omega^i \mathcal{N}( \mathbf{x} ; \bar{\mathbf{x}}^i, \mathbf{\Sigma}^i )  \f$
	 *
	 *  Where the number of modes N is the size of CPose3DPDFSOG::m_modes. Angles are always in radians.
	 *
	 *  See MRML::CPose3DPDF for more details.
     *
	 * \sa CPose3DPDF
	 */
	class CPose3DPDFSOG : public CPose3DPDF
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CPose3DPDFSOG )

	protected:
		/** Assures the symmetry of the covariance matrix (eventually certain operations in the math-coprocessor lead to non-symmetric matrixes!)
		  */
		void  assureSymmetry();

	 public:
		/** Default constructor
		  * \param nModes The initial size of CPose3DPDFSOG::m_modes
		  */
		CPose3DPDFSOG( size_t nModes = 1 );

		/** The struct for each mode:
		 */
		struct TGaussianMode
		{
			TGaussianMode()
			{
				w = 1;
			}

			CPose3DPDFGaussian		val;

			/** The (linear) weight
			  */
			double		w;
		};

		/** Access directly to this array for modify the modes as desired.
		  *  Note that no weight can be zero!!
		  */
		std::deque<TGaussianMode>	m_modes;

		 /** Returns an estimate of the point, (the mean, or mathematical expectation of the PDF)
		  */
		CPose3D  getEstimatedPose() const;

		 /** Returns an estimate of the pose covariance matrix (a 6x6 matrix for the variables x,y,z,yaw,pitch,roll)
		  */
		CMatrix  getEstimatedCovariance() const;

		/** Normalize the weights in m_modes such as they sum up to 1.
		  */
		void  normalizeWeights();

		/** Copy operator, translating if necesary (for example, between particles and gaussian representations)
		  */
		void  copyFrom(const CPose3DPDF &o);

		/** Save the density to a text file, with the following format:
		  *  There is one row per Gaussian "mode", and each row contains 10 elements:
		  *   - w (The linear weight)
		  *   - x_mean (gaussian mean value)
		  *   - y_mean (gaussian mean value)
		  *   - x_mean (gaussian mean value)
		  *   - yaw_mean (gaussian mean value, in radians)
		  *   - pitch_mean (gaussian mean value, in radians)
		  *   - roll_mean (gaussian mean value, in radians)
		  *   - C11,C22,C33,C44,C55,C66 (Covariance elements)
		  *   - C12,C13,C14,C15,C16 (Covariance elements)
		  *   - C23,C24,C25,C25 (Covariance elements)
		  *   - C34,C35,C36 (Covariance elements)
		  *   - C45,C46 (Covariance elements)
		  *   - C56 (Covariance elements)
		  *
		 */
		void  saveToTextFile(char *file) const;

		/** This can be used to convert a PDF from local coordinates to global, providing the point (newReferenceBase) from which
		  *   "to project" the current pdf. Result PDF substituted the currently stored one in the object.
		  */
		void  changeCoordinatesReference(const CPose3D &newReferenceBase );

		/** Bayesian fusion of two pose distributions, then save the result in this object (WARNING: Currently p1 must be a MRML::CPose3DPDFSOG object and p2 a MRML::CPose3DPDFSOG object)
		  */
		void  bayesianFusion( CPose3DPDF &p1, CPose3DPDF &p2 );

		/** Draws a single sample from the distribution
		  */
		void  drawSingleSample( CPose3D &outPart ) const;

		/** Draws a number of samples from the distribution, and saves as a list of 1x6 vectors, where each row contains a (x,y,z,yaw,pitch,roll) datum.
		  */
		void  drawManySamples( size_t N, std::vector<vector_float> & outSamples ) const;

		/** Returns a new PDF such as: NEW_PDF = (0,0,0) - THIS_PDF
		  */
		void  inverse(CPose3DPDF &o) const;

	}; // End of class def.

} // End of namespace

#endif
