/*---------------------------------------------------------------
	FILE: CPoint3DPDFSOG.h
	USE: See above.

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef CPointPDFSOG_H
#define CPointPDFSOG_H

#include <MRPT/MRML/CPointPDF.h>
#include <MRPT/MRML/CPointPDFGaussian.h>
#include <MRPT/UTILS/CMatrix.h>


namespace UTILS
{
	namespace OPENGL
	{
		class CSetOfObjects;
	}
}

namespace MRML
{
	/** Declares a class that represents a Probability Density function (PDF) of a 3D point \f$ p(\mathbf{x}) = [x ~ y ~ z ]^t \f$.
	 *   This class implements that PDF as the following multi-modal Gaussian distribution:
	 *
	 * \f$ p(\mathbf{x}) = \sum\limits_{i=1}^N \omega^i \mathcal{N}( \mathbf{x} ; \bar{\mathbf{x}}^i, \mathbf{\Sigma}^i )  \f$
	 *
	 *  Where the number of modes N is the size of CPointPDFSOG::m_modes
	 *
	 *  See MRML::CPointPDF for more details.
     *
	 * \sa CPointPDF, CPosePDF,
	 */
	class CPointPDFSOG : public CPointPDF
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CPointPDFSOG )

	protected:
		/** Assures the symmetry of the covariance matrix (eventually certain operations in the math-coprocessor lead to non-symmetric matrixes!)
		  */
		void  assureSymmetry();

	 public:
		/** Default constructor
		  * \param nModes The initial size of CPointPDFSOG::m_modes
		  */
		CPointPDFSOG( size_t nModes = 1 );

		/** The struct for each mode:
		 */
		struct TGaussianMode
		{
			TGaussianMode()
			{
				w = 1;
			}

			CPointPDFGaussian		val;

			/** The (linear) weight
			  */
			double		w;
		};

		/** Access directly to this array for modify the modes as desired.
		  *  Note that no weight can be zero!!
		  */
		std::deque<TGaussianMode>	m_modes;

		 /** Returns an estimate of the point, (the mean, or mathematical expectation of the PDF)
		  */
		CPoint3D  getEstimatedPoint() const;

		 /** Returns an estimate of the pose covariance matrix (3x3 cov.matrix  for x,y,z variables)
		  */
		CMatrix  getEstimatedCovariance() const;

		/** Normalize the weights in m_modes such as they sum up to 1.
		  */
		void  normalizeWeights();

		/** Copy operator, translating if necesary (for example, between particles and gaussian representations)
		  */
		void  copyFrom(const CPointPDF &o);

		/** Save the density to a text file, with the following format:
		  *  There is one row per Gaussian "mode", and each row contains 10 elements:
		  *   - w (The weight)
		  *   - x_mean (gaussian mean value)
		  *   - y_mean (gaussian mean value)
		  *   - x_mean (gaussian mean value)
		  *   - C11 (Covariance elements)
		  *   - C22 (Covariance elements)
		  *   - C33 (Covariance elements)
		  *   - C12 (Covariance elements)
		  *   - C13 (Covariance elements)
		  *   - C23 (Covariance elements)
		  *
		 */
		void  saveToTextFile(char *file) const;

		/** This can be used to convert a PDF from local coordinates to global, providing the point (newReferenceBase) from which
		  *   "to project" the current pdf. Result PDF substituted the currently stored one in the object.
		  */
		void  changeCoordinatesReference(const CPose3D &newReferenceBase );

		/** Bayesian fusion of two pose distributions, then save the result in this object (WARNING: Currently p1 must be a MRML::CPointPDFSOG object and p2 a MRML::CPointPDFSOG object)
		  */
		void  bayesianFusion( CPointPDF &p1, CPointPDF &p2 );

		/** Returns a 3D object representation of all the ellipses in the SoG.
		  */
		void  getAs3DObject( UTILS::OPENGL::CSetOfObjects	&outObj ) const;

	}; // End of class def.

} // End of namespace

#endif
