/*---------------------------------------------------------------
	FILE: CObservationStereoImages.h
	USE: See doc above

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef CObservationStereoImages_H
#define CObservationStereoImages_H

#include <MRPT/UTILS/CSerializable.h>
#include <MRPT/UTILS/CMRPTImage.h>
#include <MRPT/MRML/CObservation.h>
#include <MRPT/MRML/CPose3D.h>
#include <MRPT/MRML/CPose2D.h>
#include <MRPT/MRML/CLandmark.h>

namespace MRML
{
	class CLandmarksMap;

	/** Declares a class derived from "CObservation" that encapsule a pair of images taken by a stereo camera.
	     The next figure illustrate the coordinates reference systems involved in this class:<br>
		 <center>
		 <img src="CObservationStereoImages_figRefSystem.png">
		 </center>
	 *
	 <br>
	 <b>NOTE:</b> The images stored in this class are supposed to be UNDISTORTED images already.<br>
	 * \sa CObservation
	 */
	class CObservationStereoImages : public CObservation
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CObservationStereoImages )

	 public:
		/** Constructor.
		 * \param iplImageLeft An OpenCV "IplImage*" object with the image to be loaded in the member "imageLeft", or NULL (default) for an empty image.
		 * \param iplImageRight An OpenCV "IplImage*" object with the image to be loaded in the member "imageRight", or NULL (default) for an empty image.
		 *
		 */
		CObservationStereoImages( void *iplImageLeft = NULL, void *iplImageRight = NULL );

		/** Destructor
		 */
		~CObservationStereoImages(  );

		 /** The pose of the LEFT camera, relative to the robot.
		  */
		CPose3D			cameraPose;

		/** The A matrix, see also:
		  * \sa MRVL::VisionUtils::buildIntrinsicParamsMatrix
		  */
		CMatrix			intrinsicParams;

		/** The pair of UNDISTORTED images.
		  */
		CMRPTImage		imageLeft, imageRight;

		/** The pose of the right camera, relative to the left one:
		  *  Note that using the conventional reference coordinates for the left
		  *   camera (x points to the right, y down), the "right" camera is situated
		  *   at position (BL, 0, 0) with yaw=pitch=roll=0, where BL is the BASELINE.
		  */
		CPose3D			rightCameraPose;

		/** The focal length of the camera, in meters (can be used among 'intrinsicParams' to determine the pixel size).
		  * (Added in version 3 of object's streaming)
		  */
		double			focalLength_meters;

		 /** Implements the virtual method in charge of finding the likelihood between this
		  *   and another observation, probably only of the same derived class. The operator
		  *   may be asymmetric.
		  *
		  * \param anotherObs The other observation to compute likelihood with.
		  * \param anotherObsPose If known, the belief about the robot pose when the other observation was taken can be supplied here, or NULL if it is unknown.
		  *
		  * \return Returns a likelihood measurement, in the range [0,1].
		  *	\exception std::exception On any error, as another observation being of an invalid class.
		  */
		float  likelihoodWith( CObservation *anotherObs, CPosePDF *anotherObsPose = NULL );

		/** If buildAuxiliarMap is called before, this will contain the landmarks-map representation of the observation, for the robot located at the origin.
		  */
		CLandmarksMap	*auxMap;

		/** This method build the map in "auxMap", only the first time this is called.
		  */
		void  buildAuxiliarMap( CLandmark::TLandmarkID fID );


	}; // End of class def.

} // End of namespace

#endif
