/*---------------------------------------------------------------
	FILE: SystemUtils.h
	USE: See description below.

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef  SystemUtils_H
#define  SystemUtils_H

#include <MRPT/UTILS/utils_defs.h>

/*---------------------------------------------------------------
	Class
  ---------------------------------------------------------------*/
namespace UTILS
{
	/** A system independent time type, it holds the the number of 100-nanosecond intervals since January 1, 1601 (UTC).
     * \sa SystemUtils::getCurrentTime, SystemUtils::timeDifference
	 */
	typedef uint64_t TTimeStamp;

	/** Here there are useful functions related with time, files,memory, and storage of vectors in files.
	 */
	namespace SystemUtils
	{
		/** Extract just the name (without extension) of a filename from a complete path plus name plus extension.
		  *  This function works for either "/" or "\" directory separators.
		  * \sa extractFileExtension
		  */
		std::string  extractFileName(const std::string& filePath);

		/** Extract the extension of a filename.
		  * \sa extractFileName
		  */
		std::string  extractFileExtension(const std::string&filePath);

		/** Test if a given file exists
		 */
		bool  fileExists(const std::string& fileName);

		/** Returns the current system time.
		 */
		UTILS::TTimeStamp  getCurrentTime( );

		/** Transform from standard "time_t" (actually a double number, it can contain fractions of seconds) to TTimeStamp.
		  */
		UTILS::TTimeStamp  time_tToTimestamp( double t );

		/** Transform from standard "time_t" to TTimeStamp.
		  */
		UTILS::TTimeStamp  time_tToTimestamp( time_t t );

		/** Retuns the time difference from t1 to t2 (positive if t2 is posterior to t1), in seconds.
          */
		double timeDifference( UTILS::TTimeStamp t1, UTILS::TTimeStamp t2 );

		/** Convert a timestamp into this textual form: YEAR/MONTH/DAY,HH:MM:SS.MMM
		 */
		std::string  dateTimeToString(UTILS::TTimeStamp t);

		/** Convert a timestamp into this textual form: YEAR/MONTH/DAY
		 */
		std::string  dateToString(UTILS::TTimeStamp t);

		/** Returns the number of seconds ellapsed from midnight in the given timestamp
		 */
		double  extractDayTimeFromTimestamp(UTILS::TTimeStamp t);

		/** Convert a timestamp into this textual form: HH:MM:SS.MMM
		 */
		std::string  timeToString(UTILS::TTimeStamp t);

		/** A useful function for debuging, which saves a std::vector into a text file (compat. with MATLAB)
		* \return Returns false on any error, true on everything OK.
		*/
		bool  vectorToTextFile( std::vector<float> &vec, const std::string &fileName );

		/** A useful function for debuging, which saves a std::vector into a text file (compat. with MATLAB)
		* \return Returns false on any error, true on everything OK.
		*/
		bool  vectorToTextFile( std::vector<double> &vec, const std::string &fileName );

		/** A useful function for debuging, which saves a std::vector into a text file (compat. with MATLAB)
		* \return Returns false on any error, true on everything OK.
		*/
		bool  vectorToTextFile( std::vector<int> &vec, const std::string &fileName );

		/** A useful function for debuging, which saves a std::vector into a text file (compat. with MATLAB)
		* \return Returns false on any error, true on everything OK.
		*/
		bool  vectorToTextFile( std::vector<size_t> &vec, const std::string &fileName );

		/** Returns the memory occupied by this process, in bytes
		  */
		unsigned long  getMemoryUsage();

		/** Creates a directory
		  * \return Returns false on any error, true on everything OK.
		  */
		bool  createDirectory( const std::string &dirName );

		/** Deletes a single file. For multiple files see deleteFiles
		  * \return Returns false on any error, true on everything OK.
		  * \sa deleteFiles
		  */
		bool deleteFile( const std::string &fileName );

		/** Delete one or more files, especified by the (optimal) path and the file name (including '?' or '*') - Use forward slash ('/') for directories for compatibility between Windows and Linux, since they will be internally traslated into backward slashes ('\') if MRPT is compiled under Windows.
		  * \sa deleteFile
		  */
		void deleteFiles(const std::string &s);


		/** Returns a lower-case version of a string.
		  * \sa toLowerCase
		  */
		std::string  toUpperCase(const std::string& str);

		/** Returns an upper-case version of a string.
		  * \sa toUpperCase
		  */
		std::string  toLowerCase(const std::string& str);

		/** Returns the MRPT compilation date
		  */
		std::string MRPT_getCompilationDate();

		/** Returns a string describing the MRPT version including the SVN number.
		  */
		std::string MRPT_getVersion();

	} // End of class def.

} // End of namespace

#endif
