/*---------------------------------------------------------------
	FILE: CMatrix.h
	USE: Declares a "matrix" class. Also a type
	  for float elements matrix is defined as "CMatrix"

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef CMATRIX_H
#define CMATRIX_H

#include <MRPT/UTILS/CSerializable.h>
#include <MRPT/UTILS/CMatrixTemplateNumeric.h>
#include <MRPT/UTILS/utils_defs.h>

namespace MRML
{
	class CPose2D;
	class CPose3D;
	class CPoint2D;
	class CPoint3D;
}

namespace UTILS
{
/**  This class is a "CSerializable" wrapper for "CMatrixTemplateNumeric<float>".
 */
class CMatrix : public CSerializable, public CMatrixTemplateNumeric<float>
{
	// This must be added to any CSerializable derived class:
	DEFINE_SERIALIZABLE( CMatrix )
public:
	/** Constructor
	  */
	CMatrix(size_t row = 3, size_t col = 3) : CMatrixTemplateNumeric<float>(row,col)
	{
	}

	/** Copy constructor
	  */
	CMatrix( const CMatrixTemplateNumeric<float> &m ) : CMatrixTemplateNumeric<float>(m)
	{
	}

	/** Copy constructor
	  */
	CMatrix( const CMatrixTemplateNumeric<double> &m ) : CMatrixTemplateNumeric<float>(0,0)
	{
		*this = m;
	}

	/** Constructor from a MRML::CPose2D, which generates a 3x1 matrix \f$ [x y \phi]^T \f$
 	  */
	CMatrix( const MRML::CPose2D &p);

	/** Constructor from a MRML::CPose6D, which generates a 6x1 matrix \f$ [x y z yaw pitch roll]^T \f$
 	  */
	CMatrix( const MRML::CPose3D &p);

	/** Constructor from a MRML::CPoint2D, which generates a 2x1 matrix \f$ [x y]^T \f$
 	  */
	CMatrix( const MRML::CPoint2D &p);

	/** Constructor from a MRML::CPoint3D, which generates a 3x1 matrix \f$ [x y z]^T \f$
 	  */
	CMatrix( const MRML::CPoint3D &p);

	/** Assignment operator for float matrixes
	*/
	void  operator = (const CMatrixTemplateNumeric<float>& m)
	{
		CMatrixTemplateNumeric<float>::operator =(m);
	}

	/** Assignment operator for double matrixes
 	  */
	void  operator = (const CMatrixTemplateNumeric<double>& m)
	{
		setSize(m.getRowCount(),m.getColCount());

		for (size_t i=0;i<m.getRowCount();i++)
			for (size_t j=0;j<m.getColCount();j++)
				(*this)(i,j) = (float)m(i,j);
	}


}; // end of class definition


} // End of namespace



#endif //__STD_MATRIX_H
