/*---------------------------------------------------------------
	FILE: CPathPlanningMethod.h
	USE: See DOXYGEN description below.

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef CPathPlanningMethod_H
#define CPathPlanningMethod_H

#include <MRPT/UTILS/CDebugOutputCapable.h>
#include <MRPT/MRML/COccupancyGridMap2D.h>
#include <MRPT/MRML/CPoint2D.h>

namespace MRML
{
	/** A virtual base class for computing the optimal path for a robot
	 *	  from a origin location to a target point. See derived classes for
	 *    implementations.
     *
	 * \sa CDebugOutputCapable
	 */
	class CPathPlanningMethod : public CDebugOutputCapable
	{
	public:
		/** Default constructor
		  */
		CPathPlanningMethod();

		/** Destructor
		 */
		virtual ~CPathPlanningMethod()
		{
		}

		/** The maximum occupancy probability to consider a cell as an obstacle, default=0.5
		  */
		float	occupancyThreshold;

		/** The minimum distance between points in the returned found path (default=0.4); Notice
		  *  that full grid resolution is used in path finding, this is only a way to reduce the
		  *  amount of redundant information to be returned.
		  */
		float	minStepInReturnedPath;

		/** This method compute the optimal path for a circular robot, in the given
		  *   occupancy grid map, from the origin location to a target point.
		  * The options and additional parameters to this method can be set with
		  *   member configuration variables.
		  *
		  * \param map		[IN] The occupancy gridmap used to the planning.
		  * \param origin	[IN] The starting pose of the robot (given as a CPose2D, CPoint2D,...), in coordinates of "map".
		  * \param target	[IN] The desired target pose for the robot (given as a CPose2D, CPoint2D,...), in coordinates of "map".
		  * \param path		[OUT] The found path, in global coordinates relative to "map".
		  * \param notFound	[OUT] Will be true if no path has been found.
		  * \param maxSearchPathLength [IN] The maximum path length to search for, in meters (-1 = no limit)
		  *
		  * \exception std::exception On any error
		  */
		virtual void  computePath(
				COccupancyGridMap2D			*map,
				CPoseOrPoint				*origin,
				CPoseOrPoint				*target,
				std::deque<MRML::CPoint2D>	&path,
				bool						&notFound,
				float						maxSearchPathLength = -1
				) = 0;


	};


} // End of namespace

#endif
