/*---------------------------------------------------------------
	FILE: CObservation.h
	USE: See above

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef COBSERVATION_H
#define COBSERVATION_H

#include <MRPT/UTILS/CSerializable.h>
#include <MRPT/UTILS/SystemUtils.h>

namespace MRML
{
	class CMetricMap;
	class CObservation;
	class CPosePDF;
	class CPose2D;
	class CPose3D;

	/** Declares a class that represents any robot's observation.
	       This is a base class for many types of sensors
		     observations. Users can add a new observation type
			 creating a new class deriving from this one.<br>
		   <b>IMPORTANT</b>: Observations doesn't include any information about the
		    robot pose beliefs, just the raw observation and, where
			aplicable, information about sensor position or
			orientation respect to robotic coordinates origin.
     *
	 * \sa CSensorialFrame, CMetricMap
	 */
	class CObservation : public CSerializable
	{
		// This must be added to any CSerializable derived class:
		DEFINE_VIRTUAL_SERIALIZABLE( CObservation )

	 public:

		 /** Constructor: It sets the initial timestamp to current time
		   */
		 CObservation();

		 /** Inserts a pure virtual method for finding the likelihood between this
		  *   and another observation, probably only of the same derived class. The operator
		  *   may be asymmetric.
		  *
		  * \param anotherObs The other observation to compute likelihood with.
		  * \param anotherObsPose If known, the belief about the robot pose when the other observation was taken can be supplied here, or NULL if it is unknown.
		  *
		  * \return Returns a likelihood measurement, in the range [0,1].
		  *	\exception std::exception On any error, as another observation being of an invalid class.
		  */
		 virtual float  likelihoodWith( CObservation *anotherObs, CPosePDF *anotherObsPose = NULL ) = 0;


		 /** This method is equivalent to:
		  * \code
		  *		map->insertObservation(this, robotPose)
		  * \endcode
		  * \param theMap The map where this observation is to be inserted: the map will be updated.
		  * \param robotPose The pose of the robot base for this observation, relative to the target metric map. Set to NULL (default) to use (0,0,0deg)
		  *
		  * \return Returns true if the map has been updated, or false if this observations
		  *			has nothing to do with a metric map (for example, a sound observation).
		  *
		  * \sa CMetricMap::insertObservation
		  */
		 bool  insertObservationInto( CMetricMap *theMap, const CPose3D *robotPose = NULL ) const;

		 /** The associated time-stamp.
		  */
		UTILS::TTimeStamp	timestamp;


	}; // End of class def.

} // End of namespace

#endif
