/*---------------------------------------------------------------
	FILE: CPose3DPDF.h

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef CPose3DPDF_H
#define CPose3DPDF_H

#include <MRPT/UTILS/CSerializable.h>
#include <MRPT/MRML/CPose3D.h>

namespace MRML
{
	/** Declares a class that represents a Probability Density Function (PDF) of a 3D pose (6D actually).
	 *   This class is just the base class for unifying many diferent
	 *    ways this PDF can be implemented.
	 *
	 *  For convenience, a pose composition is also defined for any
	 *    PDF derived class, changeCoordinatesReference, in the form of a method rather than an operator.
     *
	 *  For a similar class for 3D points (without attitude), see CPointPDF
	 *
	 * \sa CPose3D, CPosePDF, CPointPDF
	 */
	class CPose3DPDF : public CSerializable
	{
		DEFINE_VIRTUAL_SERIALIZABLE( CPose3DPDF )

	 public:
		/** Returns the entropy of the estimated pose covariance matrix.
		  */
		double  getEstimatedCovarianceEntropy();

		 /** Returns an estimate of the pose, (the mean, or mathematical expectation of the PDF)
		  */
		virtual CPose3D  getEstimatedPose() const = 0 ;

		 /** Returns an estimate of the pose covariance matrix (6x6 cov.matrix for x,y,z,yaw,pitch,and roll variables)
		  */
		virtual CMatrix  getEstimatedCovariance() const = 0;

		/** Copy operator, translating if necesary (for example, between particles and gaussian representations)
		  */
		virtual void  copyFrom(const CPose3DPDF &o) = 0;

		/** Save the PDF to a text file. See derived classes for more information about the format of generated files.
		 */
		virtual void  saveToTextFile(char *file) const = 0;

		/** This can be used to convert a PDF from local coordinates to global, providing the point (newReferenceBase) from which
		  *   "to project" the current pdf. Result PDF substituted the currently stored one in the object.
		  */
		virtual void  changeCoordinatesReference( const CPose3D &newReferenceBase ) = 0;

		/** Draws a single sample from the distribution
		  */
		virtual void  drawSingleSample( CPose3D &outPart ) const = 0;

		/** Draws a number of samples from the distribution, and saves as a list of 1x6 vectors, where each row contains a (x,y,z,yaw,pitch,roll) datum.
		  */
		virtual void  drawManySamples( size_t N, std::vector<vector_float> & outSamples ) const = 0;

		/** Bayesian fusion of two pose distributions, then save the result in this object (WARNING: Currently only distributions of the same class can be fused! eg, gaussian with gaussian,etc)
		  */
		virtual void  bayesianFusion( CPose3DPDF &p1, CPose3DPDF &p2 )  = 0 ;

		/** Returns a new PDF such as: NEW_PDF = (0,0,0) - THIS_PDF
		  */
		virtual void  inverse(CPose3DPDF &o) const = 0;


	}; // End of class def.

} // End of namespace

#endif
