/*---------------------------------------------------------------
	FILE: CHMapNode.h
	USE: See doxygen doc.

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef CHMapNode_H
#define CHMapNode_H

#include <MRPT/MRML/CSensorialFrame.h>
#include <MRPT/UTILS/CSerializable.h>
#include <MRPT/UTILS/CPropertiesValuesList.h>
#include <MRPT/UTILS/CTypeSelector.h>

namespace MRML
{
	class CHierarchicalMap;
	class CHMapArc;

	/** A class for representing a node in a hierarchical map. Each node may represent
	 *    different things, which is determined by its type (see getType())
	 *
	 * \sa CHierarchicalMap,CHMapArc
	 */
	class CHMapNode : public CSerializable
	{
		friend class CHierarchicalMap;
		friend class CHierarchicalMapPartition;
		friend class CHMapArc;

		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CHMapNode )

	public:
		/** The type of the IDs of nodes.
		  */
		typedef	uint64_t THMapNodeID;

	protected:
		/** An unique identifier for the node: it is randomly generated at construction or read from stream when loaded.
		  */
		THMapNodeID				ID;

		/** The list of all arcs from/to this node:
		  */
		std::deque<CHMapArc*>	arcs;

		/** Event handler for arc destruction: It should be only called for arcs from/to this node, altought other case must be handled without effects.
		  */
		void  onArcDestruction(CHMapArc	*arc);

		/** Event handler for arc addition: It should be only called for arcs from/to this node, altought other case must be handled without effects.
		  */
		void  onArcAddition(CHMapArc	*arc);

		/** The hierarchical graph in which this object is into.
		  */
		CHierarchicalMap	*parent;

	public:
		/** Default constructor
		  */
		CHMapNode(CHierarchicalMap	*parent = NULL);

		/** Destructor
		 */
		virtual ~CHMapNode();

		/** The annotations of the node, see the general description of the class for possible properties and values.
		  */
		CPropertiesValuesList	annotations;

		/** The type of the node, the possibilities are:
		  *		- Place
		  *		- Area
		  *		- TopologicalMap
		  *		- Object
		  */
		CTypeSelector			nodeType;

		/** Reads the ID of the node (read-only property)
		  */
		THMapNodeID getID();

		/** The label of the node, only for display it to the user.
		  */
		std::string		label;

		/** Returns the level of this node in the hierarchy of arcs "arcType_Belongs", where level=0 is the ground level, 1=its parents, etc.
		  */
		unsigned int getLevelInTheHierarchy();

		/** Returns the number of arcs starting from/ending into this node.
		  */
		unsigned int getRelatedArcsCount();

		/** Returns a list with the arcs from/to this node.
		  */
		std::deque<CHMapArc*>  getArcs()
		{
			return arcs;
		}

		/** Returns a list with the arcs from/to this node.
		  */
		const std::deque<CHMapArc*>  getArcs() const
		{
			return arcs;
		}

	}; // End of class def.

} // End of namespace

#endif
