/*---------------------------------------------------------------
	FILE: CActionObservationSequence.h
	USE: See below.

   Part of the MRPT Library
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/
#ifndef CActionObservationSequence_H
#define CActionObservationSequence_H

#include <MRPT/MRML/CPose2D.h>
#include <MRPT/MRML/CSensorialFrame.h>
#include <MRPT/MRML/CActionCollection.h>

namespace MRML
{

	/** This class stores a sequence of actions and observations. There is a
	 *   sequence of objects, where each one can be of one type:
	 *		- An action:	Implemented as a CAction object, the actuation of the robot.
	 *		- Observations: Implemented as a CSensorialFrame, refering to a set of robot observations from the same pose.
	 *
	 *	This class is used to store/load files in the ".rawlog" format. Two options exists:
	 *		- To load/store a CActionObservationSequence object with the "<<" and ">>" operators.
	 *		- Or to load it from a file directly containing "CSensorialFrame" and "CPose2D" objects.  This is achieved with the "loadFromRawLogFile" method.
	 *
	 * \sa CSensorialFrame, CPose2D
	 */
	class CActionObservationSequence : public CSerializable
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CActionObservationSequence )

	public:
		/** Default constructor:
		  */
		CActionObservationSequence();

		/** Copy operator:
		  */
		void  operator =(const CActionObservationSequence &o);

		/** Destructor:
		  */
		virtual ~CActionObservationSequence();

		/** Clear the sequence.
		  */
		void  clear();

		/** Add an action to the sequence: a collection of just one element is created.
		  *   The object is duplicated, so the original one can be free if desired.
		  */
		void  addAction( CAction &action );

		/** Add a set of actions to the sequence; the object is duplicated, so the original
		  *  one can be free if desired.
		  */
		void  addActions( CActionCollection &action );

		/** Add a set of observations to the sequence; the object is duplicated, so the original
		  *  one can be free if desired.
		  */
		void  addObservations( CSensorialFrame &observations );

		/** Load the contents from a file containing one of these possibilities:
		  *		- A "CActionObservationSequence" object.
		  *		- Directly "CSensorialFrame" and "CActionCollection" objects. In this case the method stops reading on EOF of an unrecogniced class name.
		  * \returns It returns false if the file does not exists.
		  */
		bool  loadFromRawLogFile( const std::string &fileName );

		/** Returns the number of actions / observations object in the sequence.
		  */
		size_t  size();

		/** Returns true if the given object is an action, or false if it is a set
		  *   of observations, where index=0 is the first object.
		  * \sa size, getAsAction, getAsObservations
		  * \exception std::exception If index is out of bounds
		  */
		bool  isAction( size_t index );

		/** Delete the action or observation stored in the given index.
		  * \exception std::exception If index is out of bounds
		  */
		void  remove( size_t index );

		/** Returns the i'th element in the sequence, as being actions, where index=0 is the first object.
		  *  If it is not an action, it returns NULL. Do neighter modify nor delete the returned pointer.
		  * \sa size, isAction, getAsObservations
		  * \exception std::exception If index is out of bounds
		  */
		CActionCollection*  getAsAction( size_t index );

		/** Returns the i'th element in the sequence, as being an action, where index=0 is the first object.
		  *  If it is not an action, it returns NULL. Do neighter modify nor delete the returned pointer.
		  * \sa size, isAction, getAsAction
		  * \exception std::exception If index is out of bounds
		  */
		CSensorialFrame*  getAsObservations( size_t index );

	//private:
		std::deque<CSerializable*>		seqOfActObs;

	}; // End of class def.


} // End of namespace

#endif

