/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CHolonomicND_H
#define CHolonomicND_H

#include "CAbstractHolonomicReactiveMethod.h"

namespace MRPTAPPS
{
  namespace RNAV
  {

	  class CLogFileRecord_ND;

	/** An implementation of the holonomic reactive navigation method "Nearness-Diagram".
	 *   The algorithm "Nearness-Diagram" was proposed in:
	 *
	 *  Nearness diagram (ND) navigation: collision avoidance in troublesome scenarios, IEEE Transactions on
	 *   Robotics and Automation, Minguez, J. and Montano, L., vol. 20, no. 1, pp. 45-59, 2004.
	 *
	 *  \sa CAbstractHolonomicReactiveMethod,CReactiveNavigationSystem
	 */
	class CHolonomicND : public CAbstractHolonomicReactiveMethod
	{
	 public:
		 /**  Initialize the parameters of the navigator, from some configuration file, or default values if set to NULL.
		   */
		 CHolonomicND( const UTILS::CConfigFileBase *INI_FILE = NULL );

		 /** This method performs the holonomic navigation itself.
		   *  \param target [IN] The relative location (x,y) of target point.
		   *  \param obstacles [IN] Distance to obstacles from robot location (0,0). First index refers to -PI direction, and last one to +PI direction. Distances can be dealed as "meters", although they are "pseudometers", see note below.
		   *  \param maxRobotSpeed [IN] Maximum robot speed, in "pseudometers/sec". See note below.
		   *  \param desiredDirection [OUT] The desired motion direction, in the range [-PI,PI]
		   *  \param desiredSpeed [OUT] The desired motion speed in that direction, in "pseudometers"/sec. (See note below)
		   *  \param logRecord [IN/OUT] A placeholder for a pointer to a log record with extra info about the execution. Set to NULL if not required. User <b>must free memory</b> using "delete logRecord" after using it.
		   *
		   *  NOTE: With "pseudometers" we refer to the distance unit in TP-Space, thus:
		   *     <br><center><code>pseudometer<sup>2</sup>= meter<sup>2</sup> + (rad · r)<sup>2</sup></code><br></center>
		   */
		 void  navigate(	MRML::CPoint2D	&target,
							vector_float	&obstacles,
							float			maxRobotSpeed,
							float			&desiredDirection,
							float			&desiredSpeed,
							CHolonomicLogFileRecord **logRecord = NULL);

		 /** The structure used to store a detected gap in obstacles.
		   */
        struct TGap
		{
                int		ini;
                int		end;
                float	entranceDistance;
                float	maxDistance;
                int		representative_sector;
        };

		typedef std::vector<TGap> TGapArray;

		/** The set of posible situations for each trajectory.
		  */
        enum TSituations
		{
                SITUATION_TARGET_DIRECTLY = 1,
                SITUATION_SMALL_GAP,
                SITUATION_WIDE_GAP,
                SITUATION_NO_WAY_FOUND
                };

		 /**  Initialize the parameters of the navigator.
		   */
		 void  initialize( const UTILS::CConfigFileBase &INI_FILE );



	 private:
		 int	last_selected_sector;

		 int  direction2sector(float a, int N);

		/** Configuration:
		  */
		float TOO_CLOSE_OBSTACLE,WIDE_GAP_SIZE_PERCENT,RISK_EVALUATION_SECTORS_PERCENT;
		float RISK_EVALUATION_DISTANCE,MAX_SECTOR_DIST_FOR_D2_PERCENT;
		float TARGET_SLOW_APPROACHING_DISTANCE;

		vector_float factorWeights;

		/**  Find gaps in the obtacles.
		  */
        void  gapsEstimator(
					vector_float		&obstacles,
					MRML::CPoint2D		&in_target,
					TGapArray			&gaps );

		/** Search the best gap.
		  */
        void  searchBestGap(
					vector_float		&in_obstacles,
					float				in_maxObsRange,
					TGapArray			&in_gaps,
					MRML::CPoint2D		&in_target,
					int					&out_selDirection,
					float				&out_selEvaluation,
					TSituations			&out_situation,
					float				&out_riskEvaluation,
					CLogFileRecord_ND	*log);

		/** Fills in the representative sector field in the gap structure:
		  */
        void  calcRepresentativeSectorForGap(
					TGap				*gap,
					MRML::CPoint2D		&target,
					vector_float		&obstacles);

		/** Evaluate each gap:
		  */
		void  evaluateGaps(
                    vector_float		&in_obstacles,
					float				in_maxObsRange,
					TGapArray			&in_gaps,
                    int					TargetSector,
                    float				TargetDist,
                    vector_float		&out_gaps_evaluation );



	};

        /** A class for storing extra information about the execution of
	 *    CHolonomicND navigation.
	 * \sa CHolonomicND, CHolonomicLogFileRecord
	 */
	class CLogFileRecord_ND : public CHolonomicLogFileRecord
	{
		DEFINE_SERIALIZABLE( CLogFileRecord_ND )

	 public:
		 /** Member data.
		   */
                vector_int				gaps_ini,gaps_end;
				vector_float			gaps_eval;
                uint32_t                selectedSector;
                float                   evaluation;
				float					riskEvaluation;
                CHolonomicND::TSituations      situation;


	};

	DEFINE_SERIALIZABLE_POST(CLogFileRecord_ND)

  }
}


#endif



