/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CStereoGrabber_Bumblebee_H
#define CStereoGrabber_Bumblebee_H


#include <mrpt/slam/CObservationVisualLandmarks.h>
#include <mrpt/slam/CObservationStereoImages.h>
#include <mrpt/vision/utils.h>

#include <mrpt/config.h>

namespace mrpt
{
	namespace vision
	{
		/** Options used when creating a bumblebee camera capture object
		  */
		struct MRPTDLLIMPEXP TCaptureOptions_bumblebee
		{
			TCaptureOptions_bumblebee();

			int	frame_width, frame_height;	//!< Capture resolution (Default: 640x480)
			bool color;						//!< Indicates if the Bumblebee camera must capture color images (Default: false -> grayscale)
			bool getRectified;				//!< Indicates if the Bumblebee camera must capture rectified images (Default: true -> rectified)
			int	framerate;					//!< Bumblebee camera frame rate (Default: 15 fps)
		};

		/** A class for grabing stereo images from a "Bumblebee" camera
		  * NOTE:
		  *		- This class is only available when compiling MRPT with "#define MRPT_HAS_BUMBLEBEE 1" in the "config.h".
		  *		- You will need the "include" and "lib" directories of the vendor's proprietary software to be included in VC++ includes path.
		  *
		  * \sa The most generic camera grabber in MRPT: mrpt::hwdrivers::CCameraSensor
		  */
		class MRPTDLLIMPEXP  CStereoGrabber_Bumblebee
		{
		protected:
			void			*m_triclops;					//!< The Triclops context (TriclopsContext)
			void			*m_flycapture;					//!< The Flycapture context (FlyCaptureContext).
			vector_byte		m_imgBuff;						//!< A buffer to store an image

			bool			m_bInitialized;					//!< If this has been correctly initiated
			unsigned int	m_resolutionX, m_resolutionY;	//!< The desired resolution

			float			m_baseline;						//!< Camera baseline
			float			m_focalLength;					//!< Camera focal length
			float			m_centerCol, m_centerRow;		//!< Camera center coordinates		

		private:
			//CMRPTImage		m_auxLeftImg, m_auxRightImg;	//!< A pair of auxiliary CMRPTImages

			void scaleImage( void* 	image, unsigned char	ucMinOut,  unsigned char	ucMaxOut );
			void convertTriclopsImageTo8BitsIplImage( void *src, void* dst );

			/** Splits a TriclopsImage (grayscale) into two separate IplImages (from the left and right cameras) (for internal use only)
			  * triclopsImage [input]. The Triclops image to split
			  * dstL [output]. The Left CMRPTImage.
			  * dstR [output]. The Right CMRPTImage.
			*/
			static void convertTriclopsImagesToIplImages( void *triclopsImage, CMRPTImage &dstL, CMRPTImage &dstR );

		public:

			TCaptureOptions_bumblebee	m_options;			//!< Bumblebee camera frame rate (Default: 15 fps)

			/** Constructor:
			*/
			CStereoGrabber_Bumblebee( int cameraIndex = 0, const TCaptureOptions_bumblebee &options = TCaptureOptions_bumblebee() );

			// Copy constructors (no copies allowed)
			CStereoGrabber_Bumblebee( const CStereoGrabber_Bumblebee& ) :
				m_bInitialized(false),
				//m_pRectImageLeft(NULL),m_pRectImageRight(NULL),
				//m_pDispImage(NULL),
				m_resolutionX(0), m_resolutionY(0)
			{
				THROW_EXCEPTION("This class cannot be copied.");
			}

			CStereoGrabber_Bumblebee & operator =( const CStereoGrabber_Bumblebee& ) { THROW_EXCEPTION("This class cannot be copied."); }

			/** Destructor
			*/
			virtual ~CStereoGrabber_Bumblebee(void);

			/** Grab stereo images, process them, and return the computed data.
			 * \param out_observation The object to be filled with sensed data. Landmarks type is "vlColor"
			 *
			 *  NOTICE: That the member "CObservationVisualLandmarks::refCameraPose" must be
			 *           set on the return of this method, since we don't know here the robot physical structure.
			 *
			 * \return false on any error, true if all go fine.
			*/
			bool  getObservation( mrpt::slam::CObservationVisualLandmarks &out_observation);

			// ******************************** FAMD ****************************************************
			/** Grab stereo images, process them, and return the computed data.
			 * \param ROI Region of interest where the data will be computed
			 * \param out_observation The object to be filled with sensed data. Landmarks type is "vlColor"
			 *
			 *  NOTICE: That the member "CObservationVisualLandmarks::refCameraPose" must be
			 *           set on the return of this method, since we don't know here the robot physical structure.
			 *
			 * \return false on any error, true if all go fine.
			*/
			bool  getObservation(
						TROI ROI,
						mrpt::slam::CObservationVisualLandmarks &out_observation);
			// ****************************** END FAMD ***************************************************

			/** Grab stereo images, and return the pair of rectified images.
			 * \param out_observation The object to be filled with sensed data.
			 *
			 *  NOTICE: (1) That the member "CObservationStereoImages::refCameraPose" must be
			 *                set on the return of this method, since we don't know here the robot physical structure.
			 *          (2) The images are already rectified.
			 *
			 * \return false on any error, true if all go fine.
			*/
			bool  getStereoObservation( mrpt::slam::CObservationStereoImages &out_observation );


			/** This is a merge method for "getImagesPairObservation" and "getObservation", which gets both kind of observations.
			  */
			bool  getBothObservation(
						mrpt::slam::CObservationVisualLandmarks	&out_observation,
						mrpt::slam::CObservationStereoImages		&out_observationStereo );

			// ******************************** FAMD ****************************************************
			/** This is a merge method for "getImagesPairObservation" and "getObservation", which gets both kind of observations.
			  * The ROI defines the Region of Interest for the Observation
			  */
			bool  getBothObservation(
						TROI								ROI,
						mrpt::slam::CObservationVisualLandmarks	&out_observation,
						mrpt::slam::CObservationStereoImages		&out_observationStereo );

			/** This is a fast merge method for "getImagesPairObservation" and "getObservation", which gets both kind of observations.
			  * The ROI defines the Region of Interest for the Observation
			  */
			bool  getBothObservation(
				vector_float						&vX,
				vector_float						&vY,
				vector_float						&vZ,
				mrpt::slam::CObservationStereoImages		&out_observationStereo );
			// ****************************** END FAMD ***************************************************


		};	// End of class

	} // End of NS
} // End of NS


#endif
