/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CCamModel_H
#define CCamModel_H

#include <mrpt/math/CMatrixTemplateNumeric.h>
#include <mrpt/math/CVectorTemplate.h>
#include <mrpt/utils/CLoadableOptions.h>
#include <mrpt/system/os.h>
#include <mrpt/vision/utils.h>
#include <mrpt/math/lightweight_geom_data.h>

namespace mrpt
{
	namespace vision
	{
		/** Structure to hold the parameters of a pinhole camera model (for use in mono-slam).
		  *  
		  * \sa CCamModel, the application <a href="http://babel.isa.uma.es/mrpt/index.php/Application:camera-calib-gui" >camera-calib-gui</a> for calibrating a camera
		 */
		struct MRPTDLLIMPEXP TCamera{
			TCamera()
			{
				dist[0]=dist[1]=dist[2]=dist[3]=dist[4]= 0;
			}

			unsigned int	nrows;
			unsigned int	ncols;
			double	cx; //!< 'x' position of the optic center over the image
			double	cy; //!< 'y' position of the optic center over the image
			double	fx; //!< the focal distance of the camera, in 'x' pixels
			double	fy; //!< the focal distance of the camera, in 'y' pixels

			double	dist[5]; //!< [k1 k2 t1 t2 0] -> k_i: parameters of radial distortion, t_i: parameters of tangential distortion (default=0)
		};

		/** This class represent the camera model for Monocular SLAM.
		 * For now, the only model is the pinhole model.
		 *
		 * - Versions:
		 *    - First version: By Antonio J. Ortiz de Galistea.
		 *    - March 2009, ...
		 *
		 * \sa CMonoSlam, , the application <a href="http://babel.isa.uma.es/mrpt/index.php/Application:camera-calib-gui" >camera-calib-gui</a> for calibrating a camera
		*/
		class MRPTDLLIMPEXP  CCamModel : public mrpt::utils::CLoadableOptions
		{
		protected:
			vision::TCamera cam;

		public:
		/** Defalult Constructor
		 */
			CCamModel();

		/** This method loads the options from a ".ini"-like file or memory-stored string list.
		 */
		void  loadFromConfigFile(
			const mrpt::utils::CConfigFileBase	&source,
			const std::string		&section);

		/** This method displays clearly all the contents of the structure in textual form, sending it to a CStream. */
		void  dumpToTextStream( CStream		&out) const;


		/** Constructor from a init file
		 */
			CCamModel(const mrpt::utils::CConfigFileBase &cfgIni);

		/** Return the rows number for a image captured by the camera.
		 */
			unsigned int  cam_nrows()const {return cam.nrows;}

		/** Return the columns number for a image captured by the camera.
		 */
			unsigned int  cam_ncols()const {return cam.ncols;}

		/** Return the 'x' position of the optic center over the image
		 */
			double  cam_cx()const {return cam.cx;}

		/** Return the 'y' position of the optic center over the image
		 */
			double  cam_cy()const {return cam.cy;}

		/** Return the first parameter of radial distortion
		 */
			double  cam_k1()const {return cam.dist[0];}

		/** Return the second parameter of radial distortion
		 */
			double  cam_k2()const {return cam.dist[1];}

		/** Return the focal distance of the camera, in 'x' PIXELS (Element (1,1) in the camera matrix)
		 */
			double  cam_fx()const {return cam.fx;}

			/** Return the focal distance of the camera, in 'y' PIXELS (Element (2,2) in the camera matrix)
		 */
			double  cam_fy()const {return cam.fy;}

		/** Jacobian for undistortion the image coordinates
		 */
			void  jacob_undistor_fm(const mrpt::vision::TPixelCoordf &uvd, math::CMatrixDouble &J_undist);

		/** Calculate the image coordinates undistorted
		 */
			void jacob_undistor(const mrpt::vision::TPixelCoordf &p, mrpt::math::CMatrixDouble &J_undist );

		/**	Return the pixel position distorted by the camera
		 */
			void  distort_a_point(const mrpt::vision::TPixelCoordf &p, mrpt::vision::TPixelCoordf &distorted_p);

		/**	Return the pixel position undistorted by the camera
		 *	The input values 'col' and 'row' will be replace for the new values (undistorted)
		 */
			void  undistort_point(const mrpt::vision::TPixelCoordf &p, mrpt::vision::TPixelCoordf &undistorted_p);


			/**	Return the (distorted) pixel position of a 3D point given in coordinates relative to the camera (+Z pointing forward, +X to the right)
			 * \sa unproject_3D_point
  		     */
			void  project_3D_point(const mrpt::math::TPoint3D &p3D, mrpt::vision::TPixelCoordf &distorted_p) const;

			/**	Return the 3D location of a point (at a fixed distance z=1), for the given (distorted) pixel position
			 * \sa project_3D_point
			 * \note Of course, there is a depth ambiguity, so the returned 3D point must be considered a direction from the camera focus, or a vector, rather than a meaninful physical point.
  		     */
			void  unproject_3D_point(const mrpt::vision::TPixelCoordf &distorted_p, mrpt::math::TPoint3D &p3D) const;

			/** Jacobian of the projection of 3D points (with distortion), as done in project_3D_point \f$ \frac{\partial h}{\partial y} \f$, evaluated at the point p3D (read below the full explanation)

			We define \f$ h = (h_x ~ h_y) \f$ as the projected point in pixels (origin at the top-left corner), 
			and \f$ y=( y_x ~ y_y ~ y_z ) \f$ as the 3D point in space, in coordinates relative to the camera (+Z pointing forwards).

			Then this method computes the 2x3 Jacobian:

			\f[
			\frac{\partial h}{\partial y} =  \frac{\partial h}{\partial u} \frac{\partial u}{\partial y}
			\f]

			With:

			\f[
			\frac{\partial u}{\partial y} = 
			\left( \begin{array}{ccc}
			 \frac{f_x}{y_z} &  0 & - y \frac{f_x}{y_z^2} \\
			 0 & \frac{f_y}{y_z} & - y \frac{f_y}{y_z^2} \\
			\end{array} \right)
			\f]

			where \f$ f_x, f_y \f$ is the focal length in units of pixel sizes in x and y, respectively. 
			And, if we define:

			\f[
			 f = 1+ 2  k_1  (u_x^2+u_y^2)
			\f]

			then:

			\f[
			\frac{\partial h}{\partial u} = 
			\left( \begin{array}{cc}
			 \frac{ 1+2 k_1 u_y^2 }{f^{3/2}}  &  -\frac{2 u_x u_y k_1 }{f^{3/2}} \\
			 -\frac{2 u_x u_y k_1 }{f^{3/2}}  & \frac{ 1+2 k_1 u_x^2 }{f^{3/2}}
			\end{array} \right)
			\f]

			\note JLBC: Added in March, 2009. Should be equivalent to Davison's WideCamera::ProjectionJacobian
			\sa project_3D_point
			*/
			void jacobian_project_with_distortion(const mrpt::math::TPoint3D &p3D, math::CMatrixDouble & dh_dy ) const;


			/* Jacobian of the unprojection of a pixel (with distortion) back into a 3D point, as done in unproject_3D_point \f$ \frac{\partial y}{\partial h} \f$, evaluated at the pixel p
			\note JLBC: Added in March, 2009. Should be equivalent to Davison's WideCamera::UnprojectionJacobian
			\sa unproject_3D_point
			*/
			void jacobian_unproject_with_distortion(const mrpt::vision::TPixelCoordf &p, math::CMatrixDouble & dy_dh ) const;

		}; // end class

	} // end namespace
} // end namespace
#endif //__CCamModel_H
