/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef metaprogramming_H
#define metaprogramming_H

#include <mrpt/utils/CSerializable.h>

namespace mrpt
{
	namespace utils
	{
		/** A set of utility objects for metaprogramming with STL algorithms.
		  */
		namespace metaprogramming
		{
			/** An object for deleting pointers (intended for STL algorithms) */
			struct ObjectDelete
			{
				template<typename T>
				void operator()(const T *ptr)
				{
					delete ptr;
				}
			};

			/** An object for clearing an object (invokes its method "clear()") given a pointer or smart-pointer, intended for being used in STL algorithms. */
			struct ObjectClear
			{
				template<typename T>
				void operator()(T ptr)
				{
					ptr->clear();
				}
			};

			/** An object for transforming between types/classes, intended for being used in STL algorithms.
			  *  Example of usage:
			  *   \code
			  *     vector_int      v1(10);  // Input
			  *     vector_double   v2(10);  // Output
			  *     std::transform(v1.begin(),v1.end(), v2.begin(), ObjectConvert<double> );
			  *   \endcode
			  */
			template <typename TARGET_TYPE>
			struct ObjectConvert
			{
				template<typename T>
				TARGET_TYPE operator()(const T &val)
				{
					return TARGET_TYPE(val);
				}
			};

			/** An object for making smart pointers unique (ie, making copies if necessary), intended for being used in STL algorithms. */
			struct ObjectMakeUnique
			{
				void operator()(CSerializablePtr &ptr)
				{
					ptr.make_unique();
				}
			};

			/** An object for making smart pointers unique (ie, making copies if necessary), intended for being used in STL algorithms. */
			struct ObjectPairMakeUnique
			{
				template <typename T>
				void operator()(T &ptr)
				{
					ptr.first.make_unique();
					ptr.second.make_unique();
				}
			};

			/** An object for making smart pointers unique (ie, making copies if necessary), intended for being used in STL algorithms. */
			struct ObjectClearUnique
			{
				void operator()(CSerializablePtr &ptr)
				{
					ptr.clear_unique();
				}
			};

			/** An object for reading objects from a stream, intended for being used in STL algorithms. */
			struct ObjectReadFromStream
			{
			private:
				CStream		*m_stream;
			public:
				ObjectReadFromStream(CStream *stream) : m_stream(stream) {  }

				// T can be CSerializablePtr, CSerializable, or any other class implementing ">>"
				template <typename T>
				void operator()(T &obj)
				{
					(*m_stream) >> obj;
				}
			};

			/** An object for writing objects to a stream, intended for being used in STL algorithms. */
			struct ObjectWriteToStream
			{
			private:
				CStream		*m_stream;
			public:
				ObjectWriteToStream(CStream *stream) : m_stream(stream) {  }

				// T can be CSerializablePtr, CSerializable, or any other class implementing "<<"
				template <typename T>
				void operator()(const T &ptr)
				{
					(*m_stream) << ptr;
				}
			};

			/** Behaves like std::copy but allows the source and target iterators to be of different types through static typecasting.
			  * \note As in std::copy, the target iterator must point to the first "slot" where to put the first transformed element, and sufficient space must be allocated in advance.
			  * \sa copy_container_typecasting
			  */
			template<typename it_src, typename it_dst>
			void copy_typecasting(it_src  first, it_src last,it_dst  target)
			{
				for (it_src i=first; i!=last ; ++i,++target)
					*target = static_cast<typename it_src::value_type>(*i);
			}

			/** Copy all the elements in a container (vector, deque, list) into a different one performing the appropriate typecasting.
			  *  The target container is automatically resized to the appropriate size, and previous contents are lost.
			  *  This can be used to assign std::vector's of different types:
			  * \code
			  *   std::vector<int>    vi(10);
			  *   std::vector<float>  vf;
			  *   vf = vi;   // Compiler error
			  *   mrpt::utils::metaprogramming::copy_container_typecasting(v1,vf);  // Ok
			  * \endcode
			  */
			template<typename src_container, typename dst_container>
			void copy_container_typecasting(const src_container &src, dst_container &trg)
			{
				trg.resize( src.size() );
				copy_typecasting(src.begin(),src.end(),trg.begin());
			}


		} // end metaprogramming
	} // End of namespace
} // end of namespace
#endif
