/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef data_association_H
#define data_association_H

#include <mrpt/utils/utils_defs.h>
#include <mrpt/poses/CPointPDFGaussian.h>

namespace mrpt
{
	namespace slam
	{
		/** Different algorithms for data association, used in
		  */
		enum TDataAssociationMethod
		{
			assocNN = 0,  	//!< Nearest-neighbor
			assocJCBB 	 	//!< JCBB: Joint Compatibility Branch & Bound [Neira, Tardos 2001]
		};

		/** The results from mrpt::slam::data_association
		  */
		struct TDataAssociationResults
		{
			/** For each landmark ID in the observations ID_obs, its association in the predictions ID_pred = associations[ID_obs]
			  * Note that not all observations may have an associated landmark in the predictions.
			  */
			std::map<size_t, size_t>   associations;
			double   distance_maha; //!< Mahalanobis distance of the best associations found.

			/** Individual mahalanobis distances between predictions (row indices) & observations (column indices).
			  *  Indices are for the appearing order in the arguments "predictions" & "observations", they are NOT landmark IDs. 
			  *  For helping in converting indices, see the members: obsidx2id, predidx2id, id2predidx, id2obsidx
			  */
			mrpt::math::CMatrixDouble	indiv_distances;
			mrpt::math::CMatrixBool		indiv_compatibility;  //!< Like "indiv_distances" but passed thru the chi2 threshold.
			vector_uint					indiv_compatibility_counts; //!< The sum of each column of indiv_compatibility, that is, the number of compatible pairings for each observation.

			std::map<size_t,size_t> obsidx2id, predidx2id;	//!< Maps from indices in the prediction/observation vectors to their corresponding IDs
			std::map<size_t,size_t> id2predidx, id2obsidx;  //!< The oposite to obsidx2id & predidx2id.

		};

		/** Computes the data-association between the prediction of a set of landmarks and their observations, all with uncertainty.
		  * Implemented methods include (see TDataAssociation)
		  *		- NN: Nearest-neighbor
		  *		- JCBB: Joint Compatibility Branch & Bound [Neira, Tardos 2001]
		  *
		  * \param predictions [IN] The list of predicted locations of landmarks/features, indexed by their ID. The 2D/3D locations are in the same coordinate framework than "observations".
		  * \param observations [IN] The list of observed locations of landmarks/features, indexed by their ID. The 2D/3D locations are in the same coordinate framework than "predictions".
		  * \param results [OUT] The output data association hypothesis, and other useful information.
		  * \param method [IN] The selected method to make the associations.
		  * \param chi2quantile [IN] The threshold for considering a match between two close Gaussians for two landmarks, in the range [0,1]. It is used to call mrpt::math::chi2inv
		  */
		void MRPTDLLIMPEXP data_association(
			const std::map<size_t, mrpt::poses::CPointPDFGaussian>  & predictions,
			const std::map<size_t, mrpt::poses::CPointPDFGaussian>  & observation,
			TDataAssociationResults   &results,
			const TDataAssociationMethod  method = assocJCBB,
			const double  chi2quantile = 0.99
			);


	} // End of namespace
} // End of namespace

#endif
