/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CRawlogXXL_H
#define CRawlogXXL_H

#include <mrpt/poses/CPose2D.h>
#include <mrpt/slam/CSensoryFrame.h>
#include <mrpt/slam/CActionCollection.h>
#include <mrpt/utils/CFileStream.h>

namespace mrpt
{
	namespace slam
	{
		DEFINE_SERIALIZABLE_PRE( CRawlogXXL )

		using namespace mrpt::utils;

		/** This class is identical to slam::CRawlog but it transparently manages the dump of objects to a temporary file in disk, so it can handle very large rawlogs that otherwise don't fit in memory (EXPERIMENTAL, DO NOT USE!!!!).
		 *   By default, this class behaves like CRawlog, until the amount of memory used by the current proccess reaches a given limit indicated in the constructor.
		 *
		 * \sa CRawlog
		 */
		class MRPTDLLIMPEXP CRawlogXXL : public mrpt::utils::CSerializable, public mrpt::utils::CUncopiable
		{
			// This must be added to any CSerializable derived class:
			DEFINE_SERIALIZABLE( CRawlogXXL )

			DECLARE_UNCOPIABLE( CRawlogXXL )

		private:
			static const size_t INVALID_POS;

			struct TObjectElementXXL
			{
				TObjectElementXXL() : obj(), positionInTheTempFile(INVALID_POS), lastAccess(INVALID_TIMESTAMP)
				{}

				CSerializablePtr	obj;
				size_t				positionInTheTempFile;
				TTimeStamp			lastAccess;
			};

			typedef	std::deque<TObjectElementXXL> TListObjects;
			TListObjects				m_seqOfActObs;	//!< The list where the objects really are in.

			mrpt::utils::CFileStream	m_tempFile;	//!< The file with the "disk-swapped" objects.

			size_t						m_objectsInMemory; //!< How many objects are actually in memory instead of the swap file.

			/** If the given objects is not in memory, bring it back from disk (and possibly send others to disk) */
			void assureElementIsLoaded(size_t index, bool swapOutOldEntries = true);

			/** Swap entry to disk */
			void swapEntryToDisk(size_t index);

			/** Swap entries out to disk if required */
			void swapEntriesToDiskIfRequired(const TTimeStamp	tnow);

		public:

			/** Whether the temporary swap file has been opened correctly. */
			bool isOK();

			/** The type of each entry in a rawlog.
			  * \sa CRawlogXXL::getType
			  */
			enum TEntryType
			{
				etSensoryFrame = 0,
				etActionCollection,
				etObservation
			};

			/** Default constructor:
			  */
			CRawlogXXL();

			/** Copy operator:
			  * \sa moveFrom
			  */
			// No need from introduction of smart pointers
			// CRawlogXXL&  operator =(const CRawlogXXL &o);

			/** Destructor:
			  */
			virtual ~CRawlogXXL();

			/** Clear the sequence of actions/observations, freeing the memory of all the objects in the list.
			  */
			void  clear();

			/** Clear the sequence of actions/observations, without deleting the objects themselves (USE ONLY IF YOU KNOW WHAT YOU DO, NORMALLY YOU'LL CALL "clear" INSTEAD).
			  */
			void  clearWithoutDelete();

			/** Add an action to the sequence: a collection of just one element is created.
			  *   The object is duplicated, so the original one can be free if desired.
			  */
			void  addAction( CAction &action );

			/** Add a set of actions to the sequence; the object is duplicated, so the original one can be free if desired.
			  * \sa addObservations, addActionsMemoryReference
			  */
			void  addActions( CActionCollection &action );

			/** Add a set of observations to the sequence; the object is duplicated, so the original one can be free if desired.
			  * \sa addActions, addObservationsMemoryReference
			  */
			void  addObservations( CSensoryFrame &observations );

			/** Add a set of actions to the sequence, using a smart pointer to the object to add.
			  * \sa addActions, addObservationsMemoryReference, addObservationMemoryReference
			  */
			void  addActionsMemoryReference( const CActionCollectionPtr &action );

			/** Add a set of observations to the sequence, using a smart pointer to the object to add.
			  * \sa addObservations, addActionsMemoryReference, addObservationMemoryReference
			  */
			void  addObservationsMemoryReference( const CSensoryFramePtr &observations );

			/** Add a single observation to the sequence, using a smart pointer to the object to add.
			  * \sa addObservations, addActionsMemoryReference
			  */
			void  addObservationMemoryReference( const CObservationPtr &observation );

			/** Load the contents from a file containing one of these possibilities:
			  *		- A "CRawlogXXL" object.
			  *		- Directly "CSensoryFrame" and "CActionCollection" objects. In this case the method stops reading on EOF of an unrecogniced class name.
			  * \returns It returns false if the file does not exists.
			  */
			bool  loadFromRawLogFile( const std::string &fileName );

			/** Returns the number of actions / observations object in the sequence.
			  */
			size_t  size();

			/** Returns the type of a given element.
			  * \sa isAction, isObservation
			  */
			TEntryType getType( size_t index ) const;

			/** Delete the action or observation stored in the given index.
			  * \exception std::exception If index is out of bounds
			  */
			void  remove( size_t index );

			/** Returns the i'th element in the sequence, as being actions, where index=0 is the first object.
			  *  If it is not a CActionCollection, it throws an exception. Do neighter modify nor delete the returned pointer.
			  * \sa size, isAction, getAsObservations, getAsObservation
			  * \exception std::exception If index is out of bounds
			  */
			CActionCollectionPtr  getAsAction( size_t index );

			/** Returns the i'th element in the sequence, as being an action, where index=0 is the first object.
			  *  If it is not an CSensoryFrame, it throws an exception. Do neighter modify nor delete the returned pointer.
			  * \sa size, isAction, getAsAction, getAsObservation
			  * \exception std::exception If index is out of bounds
			  */
			CSensoryFramePtr  getAsObservations( size_t index );

			/** Returns the i'th element in the sequence, being its class whatever.
			  * \sa size, isAction, getAsAction, getAsObservations
			  * \exception std::exception If index is out of bounds
			  */
			CSerializablePtr    getAsGeneric( size_t index );

			/** Returns the i'th element in the sequence, as being an observation, where index=0 is the first object.
			  *  If it is not an CObservation, it throws an exception. Do neighter modify nor delete the returned pointer.
			  *  This is the proper method to obtain the objects stored in a "only observations"-rawlog file (named "format #2" above.
			  * \sa size, isAction, getAsAction
			  * \exception std::exception If index is out of bounds
			  */
			CObservationPtr  getAsObservation( size_t index );

			/** A normal iterator, plus the extra methods to determine the type of each entry in the sequence.
			  */
			class iterator
			{
			protected:
				friend class mrpt::slam::CRawlogXXL;

				//TListObjects::iterator		m_it;
				size_t						m_index; // Within the container
				mrpt::slam::CRawlogXXL*		m_parent;

			public:
				iterator(const size_t index, const CRawlogXXL *parent) : m_index(index), m_parent(const_cast<mrpt::slam::CRawlogXXL*>(parent))  {}
				virtual ~iterator() {}

				iterator & operator = (const iterator& o)
				{
					m_index = o.m_index;
					m_parent = o.m_parent;
					return *this;
				}

				bool operator == (const iterator& o) {  return m_index == o.m_index; }
				bool operator != (const iterator& o) {  return m_index != o.m_index; }

				CSerializablePtr operator *() const
				{
					// Assure it's on memory, not swapped in disk!
					m_parent->assureElementIsLoaded(m_index);
					return m_parent->m_seqOfActObs[m_index].obj;
				}

				iterator operator ++(int) {  m_index++; return *this; }
				iterator operator --(int) {  m_index--; return *this; }

				TEntryType getType() const
				{
					CSerializablePtr	o = this->operator*();

					if ( o->GetRuntimeClass()->derivedFrom( CLASS_ID(CObservation) ) )
						return etObservation;
					else if ( o->GetRuntimeClass()->derivedFrom( CLASS_ID(CSensoryFrame) ) )
						return etSensoryFrame;
					else
						return etActionCollection;
				}

				//static iterator erase( TListObjects& lst, const iterator &it) { return lst.erase(it.m_it); }
			};

			/** A normal const iterator, plus the extra methods to determine the type of each entry in the sequence.
			  */
			class const_iterator
			{
			protected:
				friend class mrpt::slam::CRawlogXXL;

				//TListObjects::iterator		m_it;
				size_t						m_index; // Within the container
				mrpt::slam::CRawlogXXL*		m_parent;

			public:
				const_iterator(const size_t index, const CRawlogXXL *parent) : m_index(index), m_parent(const_cast<mrpt::slam::CRawlogXXL*>(parent))  {}
				virtual ~const_iterator() {}

				const_iterator & operator = (const const_iterator& o)
				{
					m_index = o.m_index;
					m_parent = o.m_parent;
					return *this;
				}

				bool operator == (const const_iterator& o) {  return m_index == o.m_index; }
				bool operator != (const const_iterator& o) {  return m_index != o.m_index; }

				const CSerializablePtr operator *() const
				{
					// Assure it's on memory, not swapped in disk!
					m_parent->assureElementIsLoaded(m_index);
					return m_parent->m_seqOfActObs[m_index].obj;
				}

				const_iterator operator ++(int) {  m_index++; return *this; }
				const_iterator operator --(int) {  m_index--; return *this; }

				TEntryType getType() const
				{
					const CSerializablePtr	o = this->operator*();

					if ( o->GetRuntimeClass()->derivedFrom( CLASS_ID(CObservation) ) )
						return etObservation;
					else if ( o->GetRuntimeClass()->derivedFrom( CLASS_ID(CSensoryFrame) ) )
						return etSensoryFrame;
					else
						return etActionCollection;
				}

				//static iterator erase( TListObjects& lst, const iterator &it) { return lst.erase(it.m_it); }
			};

			const_iterator begin() const { return const_iterator(0,this ); }
			iterator begin() { return iterator( 0,this ); }
			const_iterator end() const { return const_iterator( m_seqOfActObs.size(),this ); }
			iterator end() { return iterator( m_seqOfActObs.size(),this ); }

			iterator remove(const iterator &it);

			/** Efficiently copy the contents from other existing object, and remove the data from the origin (after calling this, the original object will have no actions/observations).
			  */
			void moveFrom( CRawlogXXL &obj);

			/** Efficiently swap the contents of two existing objects.
			  */
			void swap( CRawlogXXL &obj);

		}; // End of class def.

	} // End of namespace
} // End of namespace

#endif
