/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CPOSE3D_H
#define CPOSE3D_H

#include <mrpt/poses/CPose.h>
#include <mrpt/math/CMatrixFixedNumeric.h>
#include <mrpt/math/CQuaternion.h>

namespace mrpt
{
namespace poses
{
	DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CPose3D, CPose )

	/** A class used to store a 3D pose.
	 *    A class used to store a 3D (6D) pose, including the 3D coordinate
	 *      point and orientation angles. It is used in many situations,
	 *      from defining a robot pose, maps relative poses, sensors,...
	 *		See introduction in documentation for the CPoseOrPoint class.
			<br>For a complete description of Points/Poses, see mrpt::poses::CPoseOrPoint<br>
	 *
	 *  For a complete description of Points/Poses, see mrpt::poses::CPoseOrPoint, or refer
	 *    to the <a href="http://babel.isa.uma.es/mrpt/index.php/2D_3D_Geometry">2D/3D Geometry tutorial</a> in the wiki.
	 *
	 *  To change the individual components of the pose, use CPose3D::setFromValues. This class assures that the internal
	 *   4x4 homogeneous coordinate matrix is always up-to-date with the "x y z yaw pitch roll" members.
	 *
	 *  Rotations in 3D can be also represented by quaternions. See mrpt::math::CQuaternion, and method CPose3D::getAsQuaternion.

<div align=center>

<table class=MsoTableGrid border=1 cellspacing=0 cellpadding=0
 style='border-collapse:collapse;border:none'>
 <tr style='height:15.8pt'>
  <td width=676 colspan=2 style='width:507.25pt;border:solid windowtext 1.0pt;
  background:#E6E6E6;padding:0cm 5.4pt 0cm 5.4pt;height:15.8pt'>
  <p   align=center style='text-align:center'>poses::CPose3D</p>
  </td>
 </tr>
 <tr style='height:15.8pt'>
  <td width=350 style='width:262.65pt;border:solid windowtext 1.0pt;border-top:
  none;padding:0cm 5.4pt 0cm 5.4pt;height:15.8pt'>
  <p   align=center style='text-align:center'>Homogeneous
  transfomation matrix</p>
  </td>
  <td width=326 style='width:244.6pt;border-top:none;border-left:none;
  border-bottom:solid windowtext 1.0pt;border-right:solid windowtext 1.0pt;
  padding:0cm 5.4pt 0cm 5.4pt;height:15.8pt'>
  <p   align=center style='text-align:center'>Spatial
  representation</p>
  </td>
 </tr>
 <tr style='height:202.65pt'>
  <td width=350 style='width:262.65pt;border:solid windowtext 1.0pt;border-top:
  none;padding:0cm 5.4pt 0cm 5.4pt;height:202.65pt'>
  <div align=center>
  <table  Table border=0 cellspacing=0 cellpadding=0 width=334
   style='width:250.65pt;border-collapse:collapse'>
   <tr style='height:16.65pt'>
    <td width=66 style='width:49.65pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.65pt'>
    <p   align=center style='text-align:center'>cycp</p>
    </td>
    <td width=99 style='width:74.15pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.65pt'>
    <p   align=center style='text-align:center'>cyspsr-sycr</p>
    </td>
    <td width=87 style='width:65.55pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.65pt'>
    <p   align=center style='text-align:center'>cyspcr+sysr</p>
    </td>
    <td width=82 style='width:61.3pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.65pt'>
    <p   align=center style='text-align:center'>x</p>
    </td>
   </tr>
   <tr style='height:17.25pt'>
    <td width=66 style='width:49.65pt;padding:0cm 5.4pt 0cm 5.4pt;height:17.25pt'>
    <p   align=center style='text-align:center'>sycp</p>
    </td>
    <td width=99 style='width:74.15pt;padding:0cm 5.4pt 0cm 5.4pt;height:17.25pt'>
    <p   align=center style='text-align:center'>syspsr+cycr</p>
    </td>
    <td width=87 style='width:65.55pt;padding:0cm 5.4pt 0cm 5.4pt;height:17.25pt'>
    <p   align=center style='text-align:center'>syspcr-cysr</p>
    </td>
    <td width=82 style='width:61.3pt;padding:0cm 5.4pt 0cm 5.4pt;height:17.25pt'>
    <p   align=center style='text-align:center'>y</p>
    </td>
   </tr>
   <tr style='height:19.65pt'>
    <td width=66 style='width:49.65pt;padding:0cm 5.4pt 0cm 5.4pt;height:19.65pt'>
    <p   align=center style='text-align:center'>-sp</p>
    </td>
    <td width=99 style='width:74.15pt;padding:0cm 5.4pt 0cm 5.4pt;height:19.65pt'>
    <p   align=center style='text-align:center'>cpsr</p>
    </td>
    <td width=87 style='width:65.55pt;padding:0cm 5.4pt 0cm 5.4pt;height:19.65pt'>
    <p   align=center style='text-align:center'>cpcr</p>
    </td>
    <td width=82 style='width:61.3pt;padding:0cm 5.4pt 0cm 5.4pt;height:19.65pt'>
    <p   align=center style='text-align:center'>z</p>
    </td>
   </tr>
   <tr style='height:11.0pt'>
    <td width=66 style='width:49.65pt;padding:0cm 5.4pt 0cm 5.4pt;height:11.0pt'>
    <p   align=center style='text-align:center'>0</p>
    </td>
    <td width=99 style='width:74.15pt;padding:0cm 5.4pt 0cm 5.4pt;height:11.0pt'>
    <p   align=center style='text-align:center'>0</p>
    </td>
    <td width=87 style='width:65.55pt;padding:0cm 5.4pt 0cm 5.4pt;height:11.0pt'>
    <p   align=center style='text-align:center'>0</p>
    </td>
    <td width=82 style='width:61.3pt;padding:0cm 5.4pt 0cm 5.4pt;height:11.0pt'>
    <p   align=center style='text-align:center'>1</p>
    </td>
   </tr>
  </table>
  </div>
  <p   align=center style='text-align:center'><span lang=EN-GB>where:</span></p>
  <p   align=center style='text-align:center'><span lang=EN-GB>cy
  = cos Yaw ;  sy = sin Yaw</span></p>
  <p   align=center style='text-align:center'><span lang=EN-GB>cp
  = cos Pitch ; sp = sin Pitch</span></p>
  <p   align=center style='text-align:center'><span lang=EN-GB>cr
  = cos Roll ; sr = sin Roll</span></p>
  </td>
  <td width=326 style='width:244.6pt;border-top:none;border-left:none;
  border-bottom:solid windowtext 1.0pt;border-right:solid windowtext 1.0pt;
  padding:0cm 5.4pt 0cm 5.4pt;height:202.65pt'>
  <p   align=center style='text-align:center'><span lang=EN-GB><img  src="CPose3D.gif"></span></p>
  </td>
 </tr>
</table>

</div>

	  *
	 * \sa CPoseOrPoint,CPoint3D
	 */
	class MRPTDLLIMPEXP CPose3D : public CPose
	{
		friend class CPose;
		friend class CPose2D;
		friend class CPoint;
		friend std::ostream MRPTDLLIMPEXP & operator << (std::ostream& o, const CPose3D& p);

		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CPose3D )

	protected:
		double	m_yaw, m_pitch, m_roll;	//!< These variables are updated every time that the object homogeneous matrix is modified (construction, loading from values, pose composition, etc )
		mutable CMatrixDouble44		m_HM;  //!< The homogeneous matrix

		/** Rebuild the homog matrix from x,y,z and the angles. */
		void  rebuildHomogeneousMatrix();

	 public:
		 /** Constructor with initilization of the pose; (remember that angles are always given in radians!)
		  */
		 CPose3D(const double& x=0,const double& y=0,const double& z=0,const double& yaw=0, const double& pitch=0, const double& roll=0);

		 /** Copy constructor.
		  */
		 CPose3D( const CPose3D &o);

		 /** Copy operator.
		  */
		 CPose3D & operator=( const CPose3D &o);

		 /** Constructor from a 4x4 homogeneous matrix: */
		 explicit CPose3D(const math::CMatrixDouble &m);

		 /** Constructor from a 4x4 homogeneous matrix: */
		 explicit CPose3D(const math::CMatrixDouble44 &m);

		 /** Constructor from a CPose2D object.
		  */
		 CPose3D(const CPose2D &);

		 /** Constructor from a CPoint3D object.
		  */
		 CPose3D(const CPoint3D &);

		 /** Constructor from lightweight object.
		  */
		 CPose3D(const mrpt::math::TPose3D &);

		 /** Returns the corresponding 4x4 homogeneous transformation matrix for the point(translation) or pose (translation+orientation).
		   * \sa getInverseHomogeneousMatrix
		   */
		 void  getHomogeneousMatrix(CMatrixDouble44 & out_HM ) const {
			 out_HM = m_HM;
		 }

		/** The operator \f$ a \oplus b \f$ is the pose compounding operator.
		   */
		 CPose3D  operator + (const CPose3D& b) const;

		/** The operator \f$ a \oplus b \f$ is the pose compounding operator.
		   */
		 CPoint3D  operator + (const CPoint3D& b) const;

		/** The operator \f$ a \oplus b \f$ is the pose compounding operator.
		   */
		 CPoint3D  operator + (const CPoint2D& b) const;

		 /** Scalar sum of components: This is diferent from poses
		  *    composition, which is implemented as "+" operators.
		  * \sa normalizeAngles
		  */
		 void addComponents(const CPose3D &p);

		 /** Rebuild the internal matrix & update the yaw/pitch/roll angles within the ]-PI,PI] range (Must be called after using addComponents)
		  * \sa addComponents
		   */
		 void  normalizeAngles();

		 /** Scalar multiplication of x,y,z,yaw,pitch & roll (angles will be wrapped to the ]-pi,pi] interval).
		   */
		 void operator *=(const double & s);

        /** Computes the spherical coordinates of a 3D point as seen from the 6D pose specified by this object.
          *  For the coordinate system see the top of this page.
          */
        void sphericalCoordinates(
            const CPoint3D &point,
            double &out_range,
            double &out_yaw,
            double &out_pitch ) const;

		/** An alternative, slightly more efficient way of doing \f$ G = P \oplus L \f$ with G and L being 3D points and P this 6D pose.  */
		void composePoint(double lx,double ly,double lz, double &gx, double &gy, double &gz) const;

		/** An alternative, slightly more efficient way of doing \f$ G = P \oplus L \f$ with G and L being 3D points and P this 6D pose.  */
		void composePoint(float lx,float ly,float lz, float &gx, float &gy, float &gz) const;

		/** An alternative, slightly more efficient way of doing \f$ G = P \oplus L \f$ with G and L being 3D points and P this 6D pose.
		  * \note local_point is passed by value to allow global and local point to be the same variable
		  */
		void composePoint(const TPoint3D local_point, TPoint3D &global_point) const;

		// These three must be declared here because the next three virtual ones hide the base methods.
		double x() const { return m_x; }  //!< Get the X coordinate
		double y() const { return m_y; }  //!< Get the Y coordinate
		double z() const { return m_z; }  //!< Get the Z coordinate

		void x(const double x_) { m_HM.get_unsafe(0,3)= m_x=x_; }  //!< Set the X coordinate
		void y(const double y_) { m_HM.get_unsafe(1,3)= m_y=y_; }  //!< Set the Y coordinate
		void z(const double z_) { m_HM.get_unsafe(2,3)= m_z=z_; }  //!< Set the Z coordinate

		void x_incr(const double Ax) { m_HM.get_unsafe(0,3)= m_x+=Ax; }  //!< Increment the X coordinate
		void y_incr(const double Ay) { m_HM.get_unsafe(1,3)= m_y+=Ay; }  //!< Increment the Y coordinate
		void z_incr(const double Az) { m_HM.get_unsafe(2,3)= m_z+=Az; }  //!< Increment the Z coordinate

		/** Set the pose from a 3D position (meters) and yaw/pitch/roll angles (radians) - This method recomputes the internal homogeneous coordinates matrix.
		  * \sa getYawPitchRoll, setYawPitchRoll
		  */
		void  setFromValues(
			const double		x0,
			const double		y0,
			const double		z0,
			const double		yaw=0,
			const double		pitch=0,
			const double		roll=0);

		/** Set the 3 angles of the 3D pose (in radians) - This method recomputes the internal homogeneous coordinates matrix.
		  * \sa getYawPitchRoll, setFromValues
		  */
		void  setYawPitchRoll(
			const double		yaw_,
			const double		pitch_,
			const double		roll_)
		{
			setFromValues(x(),y(),z(),yaw_,pitch_,roll_);
		}

		/** Returns the three angles (yaw, pitch, roll), in radians, from the homogeneous matrix.
		  * \sa setFromValues, yaw, pitch, roll
		  */
		void  getYawPitchRoll( double &yaw, double &pitch, double &roll );

		double yaw() const { return m_yaw; }  //!< Get the YAW angle (in radians)  \sa setFromValues
		double pitch() const { return m_pitch; }  //!< Get the PITCH angle (in radians) \sa setFromValues
		double roll() const { return m_roll; }  //!< Get the ROLL angle (in radians) \sa setFromValues

		/** The euclidean distance between two poses taken as two 6-length vectors (angles in radians).
		  */
		double distanceEuclidean6D( const CPose3D &o ) const;

		/** Returns a 1x6 vector with [x y z yaw pitch roll] */
		void getAsVector(vector_double &v) const;

		/** Returns the quaternion associated to the rotation of this object (NOTE: XYZ translation is ignored)
		  * \f[ \mathbf{q} = \left( \begin{array}{c} \cos (\phi /2) \cos (\theta /2) \cos (\psi /2) +  \sin (\phi /2) \sin (\theta /2) \sin (\psi /2) \\ \sin (\phi /2) \cos (\theta /2) \cos (\psi /2) -  \cos (\phi /2) \sin (\theta /2) \sin (\psi /2) \\ \cos (\phi /2) \sin (\theta /2) \cos (\psi /2) +  \sin (\phi /2) \cos (\theta /2) \sin (\psi /2) \\ \cos (\phi /2) \cos (\theta /2) \sin (\psi /2) -  \sin (\phi /2) \sin (\theta /2) \cos (\psi /2) \\ \end{array}\right) \f]
		  * With : \f$ \phi = roll \f$,  \f$ \theta = pitch \f$ and \f$ \psi = yaw \f$.
		  */
		void getAsQuaternion(mrpt::math::CQuaternionDouble &q) const;

	}; // End of class def.


	std::ostream MRPTDLLIMPEXP  & operator << (std::ostream& o, const CPose3D& p);

	/** Unary - operator: return the inverse pose "-p" (Note that is NOT the same than a pose with negative x y z yaw pitch roll) */
	CPose3D MRPTDLLIMPEXP operator -(const CPose3D &p);

	bool MRPTDLLIMPEXP operator==(const CPose3D &p1,const CPose3D &p2);
	bool MRPTDLLIMPEXP operator!=(const CPose3D &p1,const CPose3D &p2);

	} // End of namespace
} // End of namespace

#endif
