/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef opengl_CSetOfTriangles_H
#define opengl_CSetOfTriangles_H

#include <mrpt/opengl/CRenderizable.h>
#include <mrpt/math/geometry.h>

namespace mrpt
{
	namespace opengl
	{
		class MRPTDLLIMPEXP CSetOfTriangles;

		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CSetOfTriangles, CRenderizable )

		/** A set of colored triangles.
		  *  This class can be used to draw any solid, arbitrarily complex object (without textures).
		  *  \sa opengl::COpenGLScene, CSetOfTexturedTriangles
		  */
		class MRPTDLLIMPEXP CSetOfTriangles : public CRenderizable
		{
			DEFINE_SERIALIZABLE( CSetOfTriangles )
		public:
			struct MRPTDLLIMPEXP TTriangle
			{
				float	x[3],y[3],z[3];
				float	r[3],g[3],b[3],a[3];
			};
		protected:
			std::vector<TTriangle>		m_triangles;
			bool						m_enableTransparency;
			mutable bool polygonsUpToDate;
			mutable std::vector<mrpt::math::TPolygonWithPlane> tmpPolygons;
			void updatePolygons() const;
		public:
			inline void clearTriangles() { m_triangles.clear();polygonsUpToDate=false; }
			inline size_t getTrianglesCount() const { return m_triangles.size(); }
			inline void getTriangle(size_t idx, TTriangle &t) const { ASSERT_(idx<m_triangles.size()); t=m_triangles[idx]; }
			inline void insertTriangle( const TTriangle &t ) { m_triangles.push_back(t);polygonsUpToDate=false; }
			template<class InputIterator>
			inline void insertTriangles(const InputIterator &begin,const InputIterator &end)	{
				m_triangles.insert(m_triangles.end(),begin,end);
				polygonsUpToDate=false;
			}
			inline void insertTriangles(const CSetOfTrianglesPtr &p)	{
				m_triangles.insert(m_triangles.end(),p->m_triangles.begin(),p->m_triangles.end());
				polygonsUpToDate=false;
			}
			inline void reserve(size_t t)	{
				m_triangles.reserve(t);
			}

			inline void enableTransparency( bool v )	{ m_enableTransparency = v; }
			virtual void setColor(double R,double G,double B,double A);
			virtual void setColor(const mrpt::utils::TColorf &c);
			virtual void setColorR(const double r);
			virtual void setColorG(const double g);
			virtual void setColorB(const double b);
			virtual void setColorA(const double a);

			/** Render
			  */
			void  render() const;
			/** Ray tracing
			  */
			virtual bool traceRay(const mrpt::poses::CPose3D &o,double &dist) const;

			//Temporary method. To be deleted whenever the class is revamped to include polygons.
			void getPolygons(std::vector<mrpt::math::TPolygon3D> &polys) const;

		private:
			/** Constructor
			  */
			CSetOfTriangles( bool enableTransparency = false ) :
				m_triangles(),
				m_enableTransparency(enableTransparency),
				polygonsUpToDate(false)
			{
			}

			/** Private, virtual destructor: only can be deleted from smart pointers */
			virtual ~CSetOfTriangles() { }

		};

	} // end namespace

} // End of namespace


#endif
