/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#ifndef opengl_CSetOfLines_H
#define opengl_CSetOfLines_H

#include <mrpt/opengl/CRenderizable.h>
#include <mrpt/math/lightweight_geom_data.h>
#include <mrpt/utils/stl_extensions.h>

namespace mrpt
{
	namespace opengl
	{
		using mrpt::math::TPoint3D;
		using mrpt::math::TSegment3D;
		class MRPTDLLIMPEXP CSetOfLines;

		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CSetOfLines, CRenderizable )

		/** A set of independent lines (or segments), one line with its own start and end positions (X,Y,Z).
		  *  \sa opengl::COpenGLScene
		  */
		class MRPTDLLIMPEXP CSetOfLines : public CRenderizable
		{
			DEFINE_SERIALIZABLE( CSetOfLines )
		protected:
			std::vector<TSegment3D> mSegments;
            float			mLineWidth;

		public:
			/** Clear the list of segments */
			inline void clear()	{
				mSegments.clear();
			}

			inline void setLineWidth(float w) {
				mLineWidth=w;
			}
			float getLineWidth() const {
				return mLineWidth;
			}

			inline void appendLine(const mrpt::math::TSegment3D &sgm)	{
				mSegments.push_back(sgm);
			}

			inline void appendLine(float x0,float y0,float z0,float x1,float y1,float z1)	{
				appendLine(TSegment3D(TPoint3D(x0,y0,z0),TPoint3D(x1,y1,z1)));
			}

			inline void appendLines(const std::vector<TSegment3D> &sgms)	{
				mSegments.reserve(mSegments.size()+sgms.size());
				mSegments.insert(mSegments.end(),sgms.begin(),sgms.end());
			}

			void resize(size_t nLines)	{
				mSegments.resize(nLines);
			}

			void reserve(size_t r)	{
				mSegments.reserve(r);
			}

			template<class T,class U>
			inline void appendLine(T p0,U p1)	{
				appendLine(p0.x,p0.y,p0.z,p1.x,p1.y,p1.z);
			}

			inline size_t getLineCount() const	{
				return mSegments.size();
			}

			void setLineByIndex(size_t index,const TSegment3D &segm);

			inline void setLineByIndex(size_t index,float x0,float y0,float z0,float x1,float y1,float z1)	{
				setLineByIndex(index,TSegment3D(TPoint3D(x0,y0,z0),TPoint3D(x1,y1,z1)));
			}

			/** Class factory  */
			inline static CSetOfLinesPtr Create(const std::vector<TSegment3D> &sgms)	{
				return CSetOfLinesPtr(new CSetOfLines(sgms));
			}

			/** Render
			  */
			void  render() const;

			//Iterator management
			typedef std::vector<TSegment3D>::const_iterator const_iterator;
			typedef std::vector<TSegment3D>::const_reverse_iterator const_reverse_iterator;
			inline const_iterator begin() const	{
				return mSegments.begin();
			}
			inline const_iterator end() const	{
				return mSegments.end();
			}
			inline const_reverse_iterator rbegin() const	{
				return mSegments.rbegin();
			}
			inline const_reverse_iterator rend() const	{
				return mSegments.rend();
			}

		private:
			/** Constructor
			  */
			CSetOfLines():mSegments(),mLineWidth(1.0)	{}
			CSetOfLines(const std::vector<TSegment3D> &sgms):mSegments(sgms),mLineWidth(1.0)	{}
			/** Private, virtual destructor: only can be deleted from smart pointers */
			virtual ~CSetOfLines() { }
		};

	} // end namespace

} // End of namespace


#endif
