/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#ifndef opengl_CPointCloud_H
#define opengl_CPointCloud_H

#include <mrpt/opengl/CRenderizable.h>
#include <mrpt/utils/CMRPTImage.h>

namespace mrpt
{
	namespace slam { class CPointsMap; }

	namespace opengl
	{
		class MRPTDLLIMPEXP CPointCloud;

		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CPointCloud, CRenderizable )


		/** A cloud of points, all with the same color or each depending on it's "z" coordinate.
		  *  \sa opengl::CPlanarLaserScan, opengl::COpenGLScene, opengl::CPointCloudColoured
		  */
		class MRPTDLLIMPEXP CPointCloud : public CRenderizable
		{
			DEFINE_SERIALIZABLE( CPointCloud )
		protected:
			bool			m_colorFromZ;
			vector_float	m_xs,m_ys,m_zs;
			float           m_pointSize; //!< By default is 1.0

		public:
			void enableColorFromZ(bool v=true) { m_colorFromZ=v; }

			vector_float & getArrayX() {return m_xs;} //!< Get a reference to the internal array of X coordinates
			vector_float & getArrayY() {return m_ys;} //!< Get a reference to the internal array of Y coordinates
			vector_float & getArrayZ() {return m_zs;} //!< Get a reference to the internal array of Z coordinates

			void setPointSize(float p) { m_pointSize=p; }  //!< By default is 1.0
			float getPointSize() const { return m_pointSize; }

			/** Empty the list of points. */
			void clear();

			/** Adds a new point to the cloud */
			void insertPoint( float x,float y, float z );

			/** Load the points from a points map
			  */
			void  loadFromPointsMap( const mrpt::slam::CPointsMap *map);

			/** Render
			  */
			void  render() const;

			/** Sets the colors used as extremes when colorFromZ is enabled. */
			void  setGradientColors( const mrpt::utils::TColorf &colorMin, const mrpt::utils::TColorf &colorMax );

		private:
			/** Constructor */
			CPointCloud();

			/** Private, virtual destructor: only can be deleted from smart pointers */
			virtual ~CPointCloud() { }

			mutable float m_z_min, m_z_max; 	//!< Buffer for min/max z coords.
			mutable bool   m_z_minmax_valid;

			mrpt::utils::TColorf	m_colorFromZ_min, m_colorFromZ_max;	//!< The colors used to interpolate when m_colorFromZ is true.
		};

	} // end namespace

} // End of namespace


#endif
