/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#ifndef CMonoSlamMotionModel_H
#define CMonoSlamMotionModel_H

#include <mrpt/utils/utils_defs.h>
#include <mrpt/utils/CConfigFileBase.h>
#include <mrpt/math/CMatrixTemplateNumeric.h>
#include <mrpt/math/CVectorTemplate.h>
#include <mrpt/math/CQuaternion.h>
#include <mrpt/monoslam/link_pragmas.h>

namespace mrpt
{
	namespace monoslam
	{
		using namespace mrpt::math;

		/** General motion model in 3D
		* Assumes a random impulse changes the velocity at each time step.
		*
		* Impulse 3D Motion Model:
		* Class to represent a constant-velocity motion model.
		* A constant-velocity motion model does not mean that the camera moves at a constant velocity over all time,
		* but that the statistical model of its motion in a timestep is that on average we expect undetermined accelerations
		* occur with a Gaussian profile. At each timestep there is a random impulse, but these are normally-distributed.
		* State vector: 13 elements
		\code
						 x
		  r              y
						 z
		  -              -
						 q0
		  q              qx
						 qy
						 qz
		  -              -
						 vx
		  v              vy
						 vz
		  -              -
						 omegax
		  omega          omegay
						 omegaz
		\endcode
		* Control vector has 3 elements, and represents a measurement of
		* acceleration if a linear accelerometer is present. Otherwise set it
		* to zero.
		\code
		 Noise vector n = V
						  Omega
		\endcode
		* Update:
		\code
		 rnew     =   r + (v + V) delta_t
		 qnew     =   q x q((omega + Omega) delta_t)
		 vnew     =   v + V
		 omeganew =   omega + Omega
		\endcode
		* Here we can use the specific dimension because is a specific class for monocular SLAM with
		* Inverse Depth Parametrization
		*/
		class MONOSLAMDLLIMPEXP CMonoSlamMotionModel
		{
		public:
			static const size_t	state_size = 13;	//!< State Vector Size "xv" (in monoslam will be 13)
			static const size_t	feature_size = 6;	//!< Feature size stored in xkk (in monoslam will be 6)

			double			sigma_lin_vel;		//!< linear velocity noise sigma in m/s^2
			double			sigma_ang_vel;		//!< angular velocity noise sigma in rad/s^2
			double			delta_t;			//!< time between frames

		public:

		/** Calculate commonly used Jacobian part dq(omega * delta_t) by domega
		*/
			void dqomegadt_by_domega(const CVectorDouble &omega, CMatrixDouble &RESdqomegadt_by_domega);

		/** Ancillary function to calculate part of Jacobian partial q / partial omega which is repeatable
		*	due to symmetry. Here omegaA is one of omegax, omegay, omegaz.
		*/
			double dq0_by_domegaA(double omegaA, double omega);

		/** Ancillary function to calculate part of Jacobian which is repeatable due
		* to symmetry. Here omegaA is one of omegax, omegay, omegaz and similarly with qA.
		*/
			double dqA_by_domegaA(double omegaA, double omega);

		/** Ancillary function to calculate part of Jacobian which is repeatable due to symmetry.
		* Here omegaB is one of omegax, omegay, omegaz and similarly with qA.
		*/
			double dqA_by_domegaB(double omegaA, double omegaB,double omega);

		/** Auxiliar ancillary function to calculate part of Jacobian which is repeatable due to symmetry.
		*/
			void dq3_dq1(mrpt::math::CQuaternion<double>& q2,CMatrixDouble &M);

		/** Auxiliar ancillary function to calculate part of Jacobian which is repeatable due to symmetry.
		*/
			void dq3_dq2(mrpt::math::CQuaternion<double>& q1,CMatrixDouble &M);

		public:
			/** Default constructor
			  */
			CMonoSlamMotionModel();

			/** Return the dimension of a feature (in monoSLAM employing Unified Inverse Depth Parametrization is 6)
			 */
			size_t get_feature_size()const {return feature_size;}

			/** Return the state dimension (in monoSLAM is 13)
			 */
			size_t get_state_size()const {return state_size;}

			/** Set the time between frames
			 */
			void  set_delta_t(const double dt) {delta_t = dt;}

			/** Read the variance of linear velocity */
			double  get_sigma_lin_vel()const {return sigma_lin_vel;}

			/** Read the variance of angular velocity */
			double  get_sigma_ang_vel()const {return sigma_ang_vel;}

			/** Read time between frames */
			double  get_delta_t()const {return delta_t;}

		/** Extract from state vector their components:
		 *	\param xv --> State Vector
		 *	\param r --> camera position
		 *	\param q --> quaternion
		 *	\param v --> linear velocity
		 *	\param w --> angular velocity
		 */
			void extract_components(CVectorDouble &xv,
								CVectorDouble &r,
								CQuaternionDouble &q,
								CVectorDouble &v,
								CVectorDouble &w);

		/** Form the state vector from their individual components
		 *	\param xv --> State Vector
		 *	\param r --> camera position
		 *	\param q --> quaternion
		 *	\param v --> linear velocity
		 *	\param w --> angular velocity
		 */
			void compose_xv(CVectorDouble &xv,
								const CVectorDouble &r,
								const CQuaternionDouble &q,
								const CVectorDouble &v,
								const CVectorDouble &w);

		/** This function predict the state vector: Prediction model (camera position, angle, linear velocity and angular velocity)
		*	\param xv is the actual state vector
		*	\param r_{new} = r + (v+V)*Delta t
		*   \param q_{new} = q * q(omega + Omega) * Delta t)
		*   \param v_{new} = v + V
		*   \param w_{new} = w + W
		*	[x y z qr qx qy qz vx vy vz wx wy wz]
		*/
			void func_fv(CVectorDouble &xv);

		/** Calculate the Jacobian 'dfv/dxv'
		\code
			Identity is a good place to start since overall structure is like this
			I       0             I * delta_t   0
			0       dqnew_by_dq   0             dqnew_by_domega
			0       0             I             0
			0       0             0             I
		\endcode
		*/
			void func_dfv_by_dxv(CVectorDouble &xv, CMatrixDouble &F);

		/** Calculate the covariance noise matrix associated to the process 'Q'
		*	It's necessary to calculate the prediction 'P(k+1|k)'
		*   Fill noise covariance matrix Pnn: this is the covariance of  the noise vector [V Omega]^T
		*	that gets added to the state.
		*	Form of this could change later, but for now assume that
		*	V and Omega are independent, and that each of their components is independent.
		*/
			void func_Q(CVectorDouble &xv,CMatrixDouble &QRES);

		/** In the case of Monocular SLAM this Jacobian is a 7x7 identity matrix
		*
		*/
			void func_dxp_by_dxv(const CVectorDouble &xv,CMatrixDouble &m);


		/** Convert coordenates from inverse detph parametrization (xc,yc,zc,theta,phi,lambda) to 3D (x,y,z) coordinates
		*	Return a matrix where each row is a feature in which:
		*	- column 0 --> X coordinate
		*	- column 1 --> Y coordinate
		*	- column 2 --> Z coordinate
        * xc,yc,zc : camera position in world coord system
		* theta,phi,lambda : fteature orientation and depth params
		*/
			CMatrixDouble  XYZTPL2XYZ(CVectorDouble &xkk);

		/** Convert covariance matrix from inverse detph parametrization diag(xc,yc,zc,theta,phi,lambda) to 3D
		*	covariance matrix diag(x,y,z).
		*/
			CMatrixDouble  pxyztpl2pxyz(CMatrixDouble &pkk, CVectorDouble &xkk);

		}; // end class
	} // end namespace
} // end namespace

#endif //__CMonoSlamMotionModel.h
