/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CMonoSlam_H
#define CMonoSlam_H

#ifndef MONO_SLAM_APP
#define MONO_SLAM_APP
#endif

#include <mrpt/math/CMatrixTemplateNumeric.h>
#include <mrpt/math/CVectorTemplate.h>
#include <mrpt/math/CQuaternion.h>
#include <mrpt/math/lightweight_geom_data.h>
#include <mrpt/vision/CCamModel.h>
#include <mrpt/vision/CFeatureExtraction.h>
#include <mrpt/bayes/CKalmanFilterCapable.h>
#include <mrpt/utils/CConfigFile.h>
#include <mrpt/utils/CMRPTImage.h>
#include <mrpt/monoslam/CMonoSlam.h>
#include <mrpt/vision/CFeature.h>
#include <mrpt/hwdrivers/CCameraSensor.h>

#include <mrpt/monoslam/CFeature.h>
#include <mrpt/monoslam/CMonoSlamMotionModel.h>
#include <mrpt/monoslam/link_pragmas.h>

namespace mrpt
{
	/** Monocular SLAM classes (experimental!)
	  */
	namespace monoslam
	{
		using namespace mrpt::math;
		using namespace mrpt::poses;
		using namespace mrpt::slam;

		/**	The main class which implements Monocular SLAM (Mono-SLAM), either from a real camera or offline data.
		 * This class just contains the algorithm implementation, it's not a GUI. For directly usable programs, see "apps/mono-slam".
		 *
		 * The basic usage would be like this:
		 *  - Set all the parameters as desired in "CMonoSlam::options" and "KF_options" (CKalmanFilterCapable::KF_options). This can be done manually or by calling CMonoSlam::loadParameters
		 *	- Setup a video source in CMonoSlam::m_camera
		 *  - Call "go_one_step" to process incoming image frames
		 *	- If desired, use the provided 2D/3D renders of the filter state in "m_progress2D" & "m_progress3D"
		 *
		 *  The configuration file (.ini-like file) may have these sections:
		 * \code
		 *  [MONOSLAM]
		 *  match_threshold_up    = 	// Top threshold for feature classification
		 *  match_threshold_down  =		// Botton threshold for feature classification
		 *  ...// All the variables in mrpt::monoslam::CMonoSlam::TOptions
		 *
		 *  [MOTION_MODEL]
		 *  sigma_lin_vel    = 6.0  // linear velocity noise in m/s^2
		 *  sigma_ang_vel    = 6.0  // angular velocity noise in rad/s^2
		 *
		 *  [KALMAN]
		 *   method = 0     // 0:Full KF, 1:Efficient KF,  2: Full IKF, 3: Efficient IKF
		 *   ... // All the variables in mrpt::bayes::CKalmanFilterCapable::TKF_options
		 *
		 *  [CAMERA_MODEL]
		 *   ....
		 *
		 * \endcode
		  *
		 *
		 *
		 *  History of changes:
		 *  <ul>
		 *   <li>2007:<br>
		 * 	The first version of this application, the corresponding classes
		 * 	and config files were developed by Antonio J. Ortiz de Galisteo
		 * 	as part of his Master Thesis at the University of Malaga (2007),
		 *  based on public sources released by Andrew Davison (Imperial College London).
		 *  </li>
		 *
		 *   <li>2009:<br>
		 * 	mono-slam updated and integrated back into MRPT by M.A. Amiri Atashgah (MAAA)
		 * 	at the University of Sharif Technology, Tehran, Iran (2009), and
		 * 	Jose Luis Blanco (University of Malaga).
		 *  </li>
		 *  </ul>
		 */
		class MONOSLAMDLLIMPEXP CMonoSlam :public mrpt::bayes::CKalmanFilterCapable
		{
			public:
				/** This is the source of images. It MUST be filled before starting execution of Mono-SLAM
				  *  This smart pointer can be created from:
				  *    - Manually and loaded from a custom INI file or programatically set parameters.
				  *    - mrpt::hwdrivers::prepareVideoSourceFromUserSelection
				  *    - mrpt::hwdrivers::prepareVideoSourceFromPanel
				  *  Before executing MonoSLAM, call m_camera->initialize()
				  */
				mrpt::hwdrivers::CCameraSensorPtr  m_camera;

				/** All the options of Mono-SLAM, in a structure which can be changed manually or saved/loaded to/from a .ini file
				  * \sa KF_options
				  */
				struct MONOSLAMDLLIMPEXP TOptions : public mrpt::utils::CLoadableOptions
				{
					/** Initilization of default parameters 	*/
					TOptions();
					/** See utils::CLoadableOptions */
					void  loadFromConfigFile( const mrpt::utils::CConfigFileBase &source, const std::string &section);
					/** See utils::CLoadableOptions  */
					void  dumpToTextStream(CStream  &out) const;

					bool	verbose;				//!< Show extra debug info to console
					bool	save_data_files;		//!< Save debug info to text files
					double	match_threshold_up;		//!< Top threshold for feature classification
					double	match_threshold_down;	//!< Botton threshold for feature classification
					double	threshold_ini;			//!< Minimum quality for a feature in the initialization
					double	threshold_new;			//!< Minimum quality for a feature in the searching of a new mark
					int		radius;					//!< Minimum radius between detected image features
					double	sigma;					//!< Sigma value in the Harris method
			        unsigned int max_num_feat_to_detect ; //!< max num of features to detect

					double	lambda_init;		//!< Initial Inverse Depth estimation the first time a feature is observed
					double	std_lambda;			//!< Inverse Depth Variance for the first time a feature is observed
					double	sigma_image_noise;	//!< image noise in pixels
				};

				TOptions options; //!< All the options of Mono-SLAM, in a structure which can be changed manually or saved/loaded to/from a .ini file \sa KF_options, loadParameters

				/** With each step, "go_one_step" will save the 2D image with the detected features, etc to this image so the external program can show it to the user.  */
				CMRPTImage					m_progress2D;

				/** With each step, "go_one_step" will save the 3D scene with the filter state to this scene object so the external program can show it to the user.  */
				opengl::COpenGLScenePtr     m_progress3D;

				CMatrixDouble         m_robotPath;          //!<  robot 3D Pose
				double                m_simTime;            //!<

				/** Constructor */
				CMonoSlam();

				/** Dtor */
				virtual ~CMonoSlam();

				/** @name The main interface methods for the final user
				   @{
				  */

				/** This is the main method from the user viewpoint in Mono-SLAM: it is passed a pair Action/Observation (action can be empty, observation must contains at least 1 image), and one full Kalman Filter step will be executed.
				*  This method does:
				*	- Grab a new image frame from the camera (See m_camera).
				*	- Search Features if we have less than necessaries
				*	- EKF Prediction Step
				*	- Matching/Tracking of Features
				*	- EKF Update Step
				*	- Show/Save Results
				*
				* \param  optional_odometry This may contain odometry, if the robot has it. Left as an empty smart pointer to just rely on images (default).
				*/
				void  go_one_step(const CActionCollectionPtr &optional_odometry = CActionCollectionPtr() );

				/** Load all the parameters for Mono-SLAM, Kalman-Filter, and the motion model.
				  * \sa saveParameters
				  */
				void  loadParameters( const mrpt::utils::CConfigFileBase &source);

				/** Save all the parameters for Mono-SLAM, Kalman-Filter, and the motion model.
				  *
				  * \sa loadParameters
				  */
				void  saveParameters( mrpt::utils::CConfigFileBase &config);

				/** Resets the filter state: the camera pose is resetted and the map is emptied.
				  */
				void resetFilterState();

				/** @} */


		protected:
			int							m_iterations_count;
			unsigned int                m_num_obj_visible;
		    CMonoSlamMotionModel		mmm;					//!< MonoSLAMMotionModel
			mrpt::vision::CCamModel		cam;					//!< Camera model
	        //unsigned int	            m_min_num_feat ;          //!< min num of features to detect

			double			prev_speed;		//speed of previous iteration (for avoid strong movements)
			CVectorDouble	prev_xv;		    //state of previous iteration

			static const size_t state_size;	//!< State Vector Size
			static const size_t feature_size; //!< Feature Vector Size

			/*Members*/

			//MAAA---------------------------------------
			//unsigned int m_type;  // JL: This is "KF_options.type"
			//bool         m_bScaleHalf,m_bGrayScaleProcessing;  // This will be controlled from outside

			mrpt::system::TTimeStamp    m_dFrameTimeStamp;
			mrpt::math::CVectorDouble   m_match;

			/** Set up by go_one_step, it contains the GRAYSCALE image grabbed from the video or live camera so it can be used in the KF methods. */
			mrpt::utils::CMRPTImage     m_frame;

			/** If given by the user, set up by go_one_step, it may contain odometry */
			mrpt::slam::CActionCollectionPtr	m_action;

			// JL: Needed in the future??
			/** The mapping between landmark IDs and indexes in the Pkk cov. matrix:  */
			//std::map<CLandmark::TLandmarkID,unsigned int>	m_IDs;
			/** The mapping between indexes in the Pkk cov. matrix and landmark IDs:  */
			//std::map<unsigned int,CLandmark::TLandmarkID>	m_IDs_inverse;

			//---------------------------------------
			//int  store_index;	    //!< Index for data saving files // --> m_iterations_count

			mrpt::monoslam::CFeaturePatch	  m_FeatPatch;

			mrpt::system::TTimeStamp	m_timeLastFrame; //!< Used to estimate FPS of the video source

			// CTicTac	 m_DeltaTimeCounter; //!< Counter used to estimate FPS of the video source
			//int	sim_or_rob; // --> Done at CCameraSensor level
			//float	 m_dt;	// --> mmm.delta_m

			//gui::CDisplayWindowPtr            wind2D; // --> Managed by the GUI or console app.
			//gui::CDisplayWindow3DPtr          wind3D; // --> Managed by the GUI or console app.

			/** Show results (features selected and tracked) over actual frame. */
			void  plot_results_2D(mrpt::math::CMatrixDouble					&zz,
											mrpt::math::CMatrixDouble		&hh,
											mrpt::math::CMatrixDouble		&h,
											mrpt::math::CMatrixDouble		&S,
											mrpt::math::CMatrixDouble		 &XYZ,
											mrpt::utils::CMRPTImage			 &framecolor,
											unsigned int					type,
											mrpt::math::CVectorDouble    	&match,
											bool							verbose,
											bool							savedatas);


			/** Show the camera and feature estimated position in a 3D world */
			void  plot_results_3D(mrpt::math::CMatrixDouble &XYZ);


			/** This function is only a short method for plot in 3D windows	*/
			/*
			void  plot3D()
			{
				if (!wind3D) wind3D = new UTILS::CDisplayWindow3D("3D WORLD MONOSLAM",960,590);
				wind3D->unlockAccess3DScene();
				world = wind3D->get3DSceneAndLock();
				wind3D->unlockAccess3DScene();
				wind3D->forceRepaint();
			}
			*/

			//MAAA--------------------------------------------------

			/** Must return the action vector u.
			  * \param out_u The action vector which will be passed to OnTransitionModel
			  */
			void OnGetAction( CVectorTemplate<KFTYPE> &out_u );

			/** Implements the transition model \f$ \hat{x}_{k|k-1} = f( \hat{x}_{k-1|k-1}, u_k ) \f$
			  * \param in_u The vector returned by OnGetAction.
			  * \param inout_x At input has \f[ \hat{x}_{k-1|k-1} \f] , at output must have \f$ \hat{x}_{k|k-1} \f$ .
			  * \param out_skip Set this to true if for some reason you want to skip the prediction step (to do not modify either the vector or the covariance). Default:false
			  */
			void OnTransitionModel(
				const CVectorTemplate<KFTYPE>& in_u,
				CVectorTemplate<KFTYPE>&       inout_x,
				bool &out_skipPrediction
				 );

			/** Implements the transition Jacobian \f$ \frac{\partial f}{\partial x} \f$
			  * \param out_F Must return the Jacobian.
			  *  The returned matrix must be \f$N \times N\f$ with N being either the size of the whole state vector or get_vehicle_size().
			  */
			void OnTransitionJacobian(
				CMatrixTemplateNumeric<KFTYPE> & out_F
				 );

			/** Implements the transition noise covariance \f$ Q_k \f$
			  * \param out_Q Must return the covariance matrix.
			  *  The returned matrix must be of the same size than the jacobian from OnTransitionJacobian
			  */
			void OnTransitionNoise(
				CMatrixTemplateNumeric<KFTYPE> & out_Q
				 );

			/** This is called between the KF prediction step and the update step to allow the application to employ the prior distribution of the system state in the detection of observations (data association), where applicable.
			  * \param out_z A \f$N \times O\f$ matrix, with O being get_observation_size() and N the number of "observations": how many observed landmarks for a map, or just one if not applicable.
			  * \param out_R A matrix of size NOxO (O being get_observation_size() and N the number of observations). It is the covariance matrix of the sensor noise for each of the returned observations. The order must correspond to that in out_z.
			  * \param out_data_association An empty vector or, where applicable, a vector where the i'th element corresponds to the position of the observation in the i'th row of out_z within the system state vector, or -1 if it is a new map element and we want to insert it at the end of this KF iteration.
			  *  This method will be called just once for each complete KF iteration.
			  *  \note Since MRPT 0.5.5, the method "OnGetObservations" returning R as a matrix is called instead from the Kalman Filter engine. However, for compatibility, a default virtual method calls this method and build the R matrix from the diagonal R2.
			  */
			void OnGetObservations(
				CMatrixTemplateNumeric<KFTYPE> &out_z,
				CMatrixTemplateNumeric<KFTYPE> &out_R,
				vector_int                     &out_data_association
				);


			/** Implements the observation prediction \f$ h_i(x) \f$ and the Jacobians \f$ \frac{\partial h_i}{\partial x} \f$ and (when applicable) \f$ \frac{\partial h_i}{\partial y_i} \f$.
			  * \param in_z This is the same matrix returned by OnGetObservations(), passed here for reference.
			  * \param in_data_association The vector returned by OnGetObservations(), passed here for reference.
			  * \param in_full If set to true, all the Jacobians and predictions must be computed at once. Otherwise, just those for the observation in_obsIdx.
			  * \param in_obsIdx If in_full=false, the row of the observation (in in_z and in_data_association) whose innovation & Jacobians are to be returned now.
			  * \param out_innov The difference between the expected observation and the real one: \f$ \tilde{y} = z - h(x) \f$. It must be a vector of length O*N (in_full=true) or O (in_full=false), with O=get_observation_size() and N=number of rows in in_z.
			  * \param out_Hx One or a vertical stack of \f$ \frac{\partial h_i}{\partial x} \f$.
			  * \param out_Hy An empty matrix, or one or a vertical stack of \f$ \frac{\partial h_i}{\partial y_i} \f$.
			  *
			  *  out_Hx must consist of 1 (in_full=false) or N (in_full=true) vertically stacked \f$O \times V\f$ matrices, and out_Hy must consist of 1 or N \f$O \times F\f$ matrices, with:
			  *  - N: number of rows in in_z (number of observed features, or 1 in general).
			  *  - O: get_observation_size()
			  *  - V: get_vehicle_size()
			  *  - F: get_feature_size()
			  */

			void OnObservationModelAndJacobians(
				const CMatrixTemplateNumeric<KFTYPE> &in_z,
				const vector_int                     &data_association,
				const bool                           &in_full,
				const int                            &in_obsIdx,
				CVectorTemplate<KFTYPE>       		 &ytilde,
				CMatrixTemplateNumeric<KFTYPE>       &Hx,
				CMatrixTemplateNumeric<KFTYPE>       &Hy ) ;

			/** If applicable to the given problem, this method implements the inverse observation model needed to extend the "map" with a new "element".
			  * \param in_z This is the same matrix returned by OnGetObservations().
			  * \param in_obsIndex The index of the observation whose inverse sensor is to be computed. It corresponds to the row in in_z where the observation can be found.
			  * \param in_idxNewFeat The index that this new feature will have in the state vector (0:just after the vehicle state, 1: after that,...). Save this number so data association can be done according to these indices.
			  * \param out_yn The F-length vector with the inverse observation model \f$ y_n=y(x,z_n) \f$.
			  * \param out_dyn_dxv The \f$F \times V\f$ Jacobian of the inv. sensor model wrt the robot pose \f$ \frac{\partial y_n}{\partial x_v} \f$.
			  * \param out_dyn_dhn The \f$F \times O\f$ Jacobian of the inv. sensor model wrt the observation vector \f$ \frac{\partial y_n}{\partial h_n} \f$.
			  *
			  *  - O: get_observation_size()
			  *  - V: get_vehicle_size()
			  *  - F: get_feature_size()
			  */
			virtual void  OnInverseObservationModel(
				const CMatrixTemplateNumeric<KFTYPE> &in_z,
				const size_t                         &in_obsIdx,
				const size_t                         &in_idxNewFeat,
				CVectorTemplate<KFTYPE>              &out_yn,
				CMatrixTemplateNumeric<KFTYPE>       &out_dyn_dxv,
				CMatrixTemplateNumeric<KFTYPE>       &out_dyn_dhn );


			/** This method is called after the prediction and after the update, to give the user an opportunity to normalize the state vector (eg, keep angles within -pi,pi range) if the application requires it.
			  */
			virtual void OnNormalizeStateVector() {normalize();};

			virtual void OnPostIteration();

			/** Must return the dimension of the "vehicle state": either the full state vector or the "vehicle" part if applicable.
				 */
			virtual size_t get_vehicle_size() const
			{
                return state_size;//
			}


			/** Must return the dimension of each observation (eg, 2 for pixel coordinates, 3 for 3D coordinates,etc).
			 */
			virtual size_t get_observation_size() const
			{
				return 2;
			}
			//------------------------------------------------------------------------------------------


  			CVectorDouble   m_hrl_i;                //!< Position of the ith feature relative to the camera expressed in the image uv coordinate system

			mrpt::vision::CFeatureList  m_featList;             //!< List of all features
			mrpt::vision::CFeatureList  m_ObservedfeatList;     //!< List of observed features

    		CMatrixDouble   m_h;                    //!<  coordinate of all features the image uv coordinate system. Note: unobserved features has -1 uv values
            CMatrixDouble   m_z;                    //!<  coordinate of Observed features in the image uv coordinate system
			CMatrixDouble   m_zz;                   //!<  coordinate of matched features in the image uv coordinate system
			CMatrixDouble   m_hh;                   //!<  coordinate of precdicted features(includes just matched ones) in the image uv system
			CMatrixDouble   m_MatchCoord;           //!<  coordinate of matched features in the image uv coordinate system based on correlation value

    		CMatrixDouble               m_Hx_Hy;                //!<  Full Jacobian matrix of all features
    		CMatrixDouble               m_HH;                   //!<  Full Jacobian matrix of all matched features
    		CMatrixDouble               m_RR;                   //!<  Full R matrix of all matched features
    		CMatrixDouble               m_SS;                   //!<  Full S matrix of all matched features
		    CPose3D                     m_robotPose;            //!<  robot 3D Pose

			/** Calculate the Jacobian dh_dxv
			 */
			void  dh_dxv(CVectorDouble &yi, CMatrixDouble &zi, CMatrixDouble &m_dh_dxv);

			/** Calculate the Jacobian dh_dy
			 */
			void  dh_dyi(CVectorDouble &yi, CMatrixDouble &zi, CMatrixDouble &m_dh_dy);

			/** Calculate the Jacobian dh_drw
			 */
			void  dh_drw(CVectorDouble &yi, CMatrixDouble &zi, CMatrixDouble &m_dh_drw);

			/** Calculate the Jacobian dh_dqwr
			 */
			void  dh_dqwr(CVectorDouble &yi, CMatrixDouble &zi, CMatrixDouble &m_dh_dqwr);

			/** Calculate the Jacobian dh_dhrl
			 */
			void  dh_dhrl(CVectorDouble &yi, CMatrixDouble &zi, CMatrixDouble &m_dh_dhrl);

			/** Calculate the Jacobian dhrl_drw
			 */
			void  dhrl_drw(CVectorDouble &yi, CMatrixDouble &m_dhrl_drw);

			/** Calculate the Jacobian dhd_dhu
			 */
			void  dhd_dhu(CMatrixDouble &zi, CMatrixDouble &m_dhd_dhu);

			/** Calculate the Jacobian dhu_dhrl
			 */
			void  dhu_dhrl(CVectorDouble &yi, CMatrixDouble &m_dhu_dhrl);

			/** Calculate the Jacobian dhrl_dqwr
			 */
			void  dhrl_dqwr(CVectorDouble &yi, CMatrixDouble &m_dhrl_dqwr);

			/** Calculate the Jacobian dhrl_dy
			 */
			void  dhrl_dy(CVectorDouble &yi, CMatrixDouble &m_dhrl_dy);


			/** Calculate and return the feature coordinate in Unified Inverse Depth Parametrization. Where h_LR_rot
			 * is the vector which join the camera optic center to the feature.
			 */
			CVectorDouble  hinv(CMatrixDouble h_LR_rot);

			/** Calculate and return the vector which join the camera optic center to the feature.
			 */
			CMatrixDouble calculate_ray(const float &col,const float &row);

			/** This function increment the covariance matrix with a new feature.
			*/
			void  addAFeatureCov_newPar(CMatrixDouble &XYZ_w, float &col, float &row);

			/** Calculate the mahalanobis distance in feature innovation (prediction - measurement)
			 */
			void  mahalanob_dist(int j_k, int i_k, int halfW_x, int halfW_y, CMatrixDouble &inv_S, CMatrixDouble &correlationScore);


		protected:

			/** This function insert a new feature to the map (state vector and covariance matrix).
			 * \param col is the feature column position in pixels over the image
			 * \param row is the feature row position in pixels over the image
			 */
			void  addNewFeatures_newPar(float &col,float &row);

			/** Calculate the feature's position and the transition matrix H = [dh_dxv  dh_dy]
			 *	Measurement, derivatives and texture prediction
			 * \param h, output. This vector contain the feature position estimation.
			 * \param H, output. This matrix is the Jacovian of measurement prediction.
			 */
			void  h_calculate(CMatrixDouble &h_Out, CMatrixDouble &H_Out);

			/** This function return the position,zi (in pixels) of a feature yi
			 */

			//returns distorted pixel coord
			void  hi(CVectorDouble &yinit, CMatrixDouble &zi, size_t indx );


			/** % Calculate the projection of yi (in the camera reference)
			 */

			//returns undistorted pixel coord
			//Project a 3D point into undistorted pixel coordinates

			void  hu(const TPoint3D &XYZ, mrpt::vision::TPixelCoordf &out_pixel );

			/** % Calculate h derivates (h = feature's position in pixels).	Hi = [dh_dxv dh_dyi]
			 * \param yi --> feature in Unified Inverse Depth Parametrization
			 * \param zi --> measurement in pixels coordinates
			 * \param i  --> feature index
			 * \param Hi --> measurement jacobian
			 */
			void  calculate_Hi(CVectorDouble &yi, CMatrixDouble &zi, unsigned int i, CMatrixDouble &Hi);

			/** Matching between actual image and feature patch	(FOR EKF)
			 \code
				input parameters
				h_predicted --> (u,v) coordinates of predicted features (is a column vector (u0,v0,u1,v1,.,un,vn))
				S_predicted --> predicted covariance innovation matrix
				H_predicted --> predicted transition matrix
				im			--> current frame

				output parameters
				h			--> (u,v) coordinates of predicted features (rounded) (is a column vector (u0,v0,u1,v1,.,un,vn))
				z			--> (u,v) coordinates of measured features  (rounded) (is a column vector (u0,v0,u1,v1,.,un,vn))
				H_matched	--> measured transition matrix
				R_matched	--> measured process noise matrix

				inout parameter
				features	--> (As input) features struct to get feature's patch
								(As output)update hits for erase bad features later
			 \endcode
			 */
			void  MSMatching(CMatrixDouble &h_predicted,  const CMatrixDouble &S_predicted,
									   CMatrixDouble &H_predicted,  mrpt::utils::CMRPTImage &im,
									   CMatrixDouble &hh,			  CMatrixDouble &zz,
									   CMatrixDouble &H_matched,	  CMatrixDouble &R_matched,
									   CFeaturePatch &features);

			/** Matching between actual image and feature patch	(FOR UKF and IEKF)
			\code
				input parameters
				h_predicted --> (u,v) coordinates of predicted features (is a column vector (u0,v0,u1,v1,.,un,vn))
				S_predicted --> predicted covariance innovation matrix
				im			--> current frame

				output parameters
				z			--> (u,v) coordinates of measured features  (rounded) (is a column vector (u0,v0,u1,v1,.,un,vn))
				R_matched	--> measured process noise matrix

				inout parameter
				features	--> (As input) features struct to get feature's patch
								(As output)update hits for erase bad features later
			\endcode
			*/
			void  MSMatching(CMatrixDouble &h_predicted,			CMatrixDouble &S_predicted,
										  mrpt::utils::CMRPTImage &im,					CMatrixDouble &hh,
										  CMatrixDouble &zz,						CMatrixDouble &R_matched,
										  CVectorDouble &match,	CFeaturePatch &features);


			/** Initialize feature employing the Harris method adding neccesary information to x(k|k) and p(k|k)
			 * \param features	--> Structure to store the visual features
			 * \param im		--> Initial image in the system
			 */
			void  InitializeNewFeature(CFeaturePatch &features, mrpt::utils::CMRPTImage &im);

			/** Try to find a image area without features and search it employing the Harris method adding neccesary information to x(k|k) and p(k|k)
			 * \param features	--> Structure to store the visual features
			 * \param im		--> Image where search more features
			 * \param h			--> Predicted features over the image
			 */
			void  SearchMoreFeature(CFeaturePatch &features, mrpt::utils::CMRPTImage &im, CMatrixDouble &h);


			/** This function lighten the ambiguity problem when measure features for Monocular SLAM when employ the EKF.
			 * \param HH		--> Measured transition matrix
			 * \param RR		--> Measured process noise matrix
			 * \param hh		--> (u,v) coordinates of predicted features (rounded) (is a column vector (u0,v0,u1,v1,.,un,vn))
			 * \param zz		--> (u,v) coordinates of measured features  (rounded) (is a column vector (u0,v0,u1,v1,.,un,vn))
			 */
			void  Ambiguity(CMatrixDouble & HH,CMatrixDouble &RR,CMatrixDouble &hh,CMatrixDouble &zz);

			/** This function lighten the ambiguity problem when measure features for Monocular SLAM when employ the IEKF or UKF.
			 * \param RR		--> Measured process noise matrix
			 * \param hh		--> (u,v) coordinates of predicted features (rounded) (is a column vector (u0,v0,u1,v1,.,un,vn))
			 * \param zz		--> (u,v) coordinates of measured features  (rounded) (is a column vector (u0,v0,u1,v1,.,un,vn))
			 * \param match		--> This vector contain the information about which features has been correctly measured and which not.
			 */
			void  Ambiguity(CMatrixDouble & RR,CMatrixDouble &hh,CMatrixDouble &zz,CVectorDouble &match);

	/********************************************   FUNCTIONS FOR EKF   ********************************************/
			/** Generic function to acces to the prediction model: fv (it must be implemented in MonoSLAMMotionModel)
			 *	\param u --> odometry information (unused parameter for MonoSLAM)
			 *	\param xv --> camera state
			 */
			void  function_fv(const CMatrixDouble &u, CVectorDouble &xv);

			/** Generic function to acces to the prediction model Jacobian, F due to the linealization process
			 * (it must be implemented in MonoSLAMMotionModel)
			 */
			void  function_F(CMatrixDouble &F_parcial);

			/** Generic function to acces to the matrix process noise, Q
			 * (it must be implemented in MonoSLAMMotionModel)
			 */
			void  function_Q(CMatrixDouble &Q);

			/** Measurement model
			 * \param h --> Vector to return the measurement prediction coordinates
			 * \param Hx_Hy --> Jacobian [Hx Hy1...Hyn]
 			 */
			void  function_h(CMatrixDouble &h, CMatrixDouble &Hx_Hy);

			/** Generic function to acces to the measurement process noise, R
			 * (it must be implemented in MonoSLAMMotionModel)
			 */
			void  function_R(CMatrixDouble &h, CMatrixDouble &R);

			/** This function normalize the state vector and covariance matrix due to the quaternion employ for orientation.
			 * If the orientation is not represented by a quaternion, this function is empty.
			 */
			void  normalize();

			/** This function delete from the state vector and covariance matrix the feature in the position i.
			 */
			void delete_feature(const unsigned int &index);

			/** Return the state size. In Monocular SLAM this value is 13
			 */
			unsigned int  get_state_size() const {return state_size;}

			/** Return the feature size. In Monocular SLAM this value is 6 due to the employ of Unified Inverse Depth Parametrization.
			 */
			size_t  get_feature_size() const {return feature_size;}

			/** Transform the Unified Inverse Depth Parametrization coordinates to XYZ cartesian coordinates.
			 */
			CMatrixDouble  XYZ6D_TO_XYZ3D(){return mmm.XYZTPL2XYZ(m_xkk);}

			/** Transform the Unified Inverse Depth Parametrization covariance matrix (6x6) to XYZ covariance (3x3).
			 */
			CMatrixDouble  pxyztpl2pxyz(){return mmm.pxyztpl2pxyz(m_pkk, m_xkk);}


	/********************************************   FUNCTIONS FOR uKF   ********************************************/
			/** Measurement model
			 * \param h --> Vector to return the measurement prediction coordinates
			 */
			void  h_UKF(CMatrixDouble &h);

			/** Visibility test for determine which features are shown in the image
			 * \param h --> Vector to return the measurement prediction coordinates
			 */
			void  visibility(CMatrixDouble &h);


	/************************************************   NEWBITS   ***************************************************/
			/** Jacobian of quaternion inverse: this is a constant matrix that doesn't
			* depend on the quaternion
			*/
			CMatrixDouble  dqbar_by_dq();	//matrix(4x4)

			/** Component Jacobians: each returns a 3x3 matrix which is the derivative
			* of the rotation matrix derived from a quaternion w.r.t. return the real element
			*/
			CMatrixDouble  dR_by_dq0(CQuaternionDouble &q);	//matrix(3x3)

			/** Component Jacobians: each returns a 3x3 matrix which is the derivative
			* of the rotation matrix derived from a quaternion w.r.t. the imaginary element x
			*/
			CMatrixDouble  dR_by_dqx(CQuaternionDouble &q);	//matrix(3x3)

			/** Component Jacobians: each returns a 3x3 matrix which is the derivative
			* of the rotation matrix derived from a quaternion w.r.t. the imaginary element y
			*/
			CMatrixDouble  dR_by_dqy(CQuaternionDouble &q);	//matrix(3x3)

			/** Component Jacobians: each returns a 3x3 matrix which is the derivative
			* of the rotation matrix derived from a quaternion w.r.t. the imaginary element z
			*/
			CMatrixDouble  dR_by_dqz(CQuaternionDouble &q);	//matrix(3x3)


			/** Auxiliary functions used by dqnorm_by_dq(). Value of diagonal element of Jacobian
			*/
			float  dqi_by_dqi(float qi, float qq);

			/** Value of off-diagonal element of Jacobian
			*/
			float  dqi_by_dqj(float qi, float qj, float qq);


			/** Normalising Jacobian
			*/
			CMatrixDouble  dqnorm_by_dq(CQuaternionDouble &q);	//matrix (4x4)

			/** Normalising a 3D vector
			 */
			CMatrixDouble  dvnorm_by_dv(CVectorDouble &v);	//matrix(3x3)

			/** Auxiliary function. User must not be used.
			 * \param q --> state vector quaternion.
			 * \param a --> (hLhatRi) Not rotate.
			 */
			CMatrixDouble  dRq_times_a_by_dq (CQuaternionDouble &q, CMatrixDouble &aMat); //matrix(3x4)

	/***********************************************************************/
	//ESTA FUNCION SOLO SIRVE PARA PROBAR LA CLASE DESDE MATLAB, BORRAR CND TERMINE LA DEPURACION

			/**	This function set the state vector value. (Employed for debug the program)
			 */

			void  set_xkk(const CVectorDouble &xkk_in){m_xkk = xkk_in;}

			/**	This function set the covariance matrix value. (Employed for debug the program)
			 */
			void  set_pkk(const CMatrixDouble &pkk_in){m_pkk = pkk_in;}

			/**	This function read the state vector value. (Employed for debug the program)
			 */
			void  get_xkk(CVectorDouble &xkk_out)
			{
				xkk_out.resize(m_xkk.size());
				xkk_out = m_xkk;
			}

			/**	This function read the covariance matrix value. (Employed for debug the program)
			 */
			void  get_pkk(CMatrixDouble &pkk_out){
				pkk_out.resize(m_pkk.getRowCount(),m_pkk.getColCount());
				pkk_out = m_pkk;
			}

		}; // end class

	} // end namespace
} // end namespace

#endif //__CCMonoSlam_H
