/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CHMHMapNode_H
#define CHMHMapNode_H

#include <mrpt/utils/safe_pointers.h>
#include <mrpt/utils/stl_extensions.h>
#include <mrpt/slam/CSensoryFrame.h>
#include <mrpt/hmtslam/HMT_SLAM_common.h>

#include <mrpt/utils/CSerializable.h>
#include <mrpt/utils/CMHPropertiesValuesList.h>
#include <mrpt/utils/CTypeSelector.h>

namespace mrpt
{
	namespace hmtslam
	{
		using namespace mrpt::slam;

		class HMTDLLIMPEXP CHierarchicalMHMap;
		class HMTDLLIMPEXP CHMHMapArc;

		DEFINE_SERIALIZABLE_PRE_CUSTOM_LINKAGE( CHMHMapNode, HMTDLLIMPEXP )

		/** A class for representing a node in a hierarchical, multi-hypothesis map.
		 *   The node itself will be considered only if some given hypothesisID matchs its own ID.
		 *
		 * \sa CHierarchicalMHMap,CHMHMapArc
		 */
		class HMTDLLIMPEXP CHMHMapNode : public mrpt::utils::CSerializable
		{
			friend class HMTDLLIMPEXP CHierarchicalMHMap;
			friend class HMTDLLIMPEXP CHierarchicalMHMapPartition;
			friend class HMTDLLIMPEXP CHMHMapArc;

			// This must be added to any CSerializable derived class:
			DEFINE_SERIALIZABLE( CHMHMapNode )

		public:
			/** The type of the IDs of nodes.
			  */
			typedef	int64_t TNodeID;

			/** The hypothesis IDs under which this node exists.
			  */
			THypothesisIDSet		m_hypotheses;

		protected:
			/** An unique identifier for the node: it is randomly generated at construction or read from stream when loaded.
			  */
			TNodeID					m_ID;

			/** The list of all arcs from/to this node:
			  */
			TArcList				m_arcs;

			/** Event handler for arc destruction: It should be only called for arcs from/to this node, altought other case must be handled without effects.
			  */
			void  onArcDestruction(CHMHMapArc	*arc);

			/** Event handler for arc addition: It should be only called for arcs from/to this node, altought other case must be handled without effects.
			  */
			void  onArcAddition(CHMHMapArc	*arc);

			/** The hierarchical graph in which this object is into.
			  */
			safe_ptr<CHierarchicalMHMap>	m_parent;

			static bool m_firstRandomize;

		public:
			/** Default constructor
			  */
			CHMHMapNode(
				CHierarchicalMHMap		*parent = NULL,
				const THypothesisIDSet	&hyps = THypothesisIDSet() );

			/** Destructor
			 */
			virtual ~CHMHMapNode();

			/** The annotations of the node, see the general description of the class for possible properties and values.
			  */
			utils::CMHPropertiesValuesList	m_annotations;

			/** The type of the node, the possibilities are:
			  *		- Place
			  *		- Area
			  *		- TopologicalMap
			  *		- Object
			  */
			utils::CTypeSelector			m_nodeType;

			/** Reads the ID of the node (read-only property)
			  */
			TNodeID getID() const;

			/** The label of the node, only for display it to the user.
			  */
			std::string		m_label;

			/** Returns the level of this node in the hierarchy of arcs "arcType_Belongs", where level=0 is the ground level, 1=its parents, etc.
			  */
			unsigned int getLevelInTheHierarchy();

			/** Returns the number of arcs starting from/ending into this node.
			  */
			unsigned int getRelatedArcsCount();

			/** Returns a list with the arcs from/to this node.
			  */
			void getArcs( TArcList &out ) const
			{
				out = m_arcs;
			}

			/** Returns a list with the arcs from/to this node existing in a given hypothesis ID.
			  */
			void getArcs( TArcList &out, const THypothesisID &hyp_id ) const;

			/** Returns a list with the arcs from/to this node existing in a given hypothesis ID and of a given type.
			  */
			void getArcs( TArcList &out, const char *arcType, const THypothesisID &hyp_id ) const;

			/** Check whether an arc exists towards the giben area */
			bool isNeighbor(const TNodeID &otherArea, const THypothesisID &hyp_id ) const;

		}; // End of class def.


		/** A map between node IDs and nodes (used in HMT-SLAM).
		  * \sa CHMTSLAM
		  */
		typedef std::map<CHMHMapNode::TNodeID,CHMHMapNode*>  TNodeList;

		typedef list_searchable<CHMHMapNode::TNodeID> TNodeIDList;

		/** A map between node IDs and nodes (used in HMT-SLAM).
		  * \sa CHMTSLAM
		  */
		typedef std::map<CHMHMapNode::TNodeID,const CHMHMapNode*>  TNodeListConst;

		typedef std::pair<CHMHMapNode::TNodeID,CHMHMapNode::TNodeID>  TPairNodeIDs;

	} // End of namespace
} // End of namespace

#endif
