/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CPOLYGON_H
#define CPOLYGON_H

#include <MRPT/UTILS/utils_defs.h>
#include <MRPT/UTILS/CSerializable.h>

namespace UTILS
{
	/** A simple 2D polygon class, mainly for checking points falling inside it.
	 */
	class CPolygon : public UTILS::CSerializable
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CPolygon )

	private:
		/** The placeholders for vertexs
		 */
		std::vector<double>	vx,vy;

		/** The "central" point coordinates (laser sensor location if applicable)
		 */
		double				cx,cy;

		/** Automatically maintained max/min coordinates of polygon.
		 */
		double			min_x,max_x,min_y,max_y;

		/* Return "true" if P is on the left side of P0-P1:
		 */
		bool	isLeft( double P0x,double P0y,double P1x,double P1y,double Px,double Py  );

	public:
		/** Constructor:
		 */
		CPolygon(double cx = 0,double  cy = 0);

		/** Add a new vertex to polygon:
		 */
		void  AddVertex(double x,double y);

		/** Methods for accessing the vertexs:
		 */
		double  GetVertex_x(size_t i) { return vx[i]; }
		double  GetVertex_y(size_t i) { return vy[i]; }

		/** Returns the vertices count in the polygon:
		  */
		size_t   verticesCount() { return vx.size(); }

		/** Set all vertices at once, not to be used normally.
		  */
		void  setAllVertices( size_t nVertices, double *xs, double *ys );

		/** Set all vertices at once, not to be used normally.
		  */
		void  setAllVertices( size_t nVertices, float *xs, float *ys );

		/** Clear the polygon, erasing all vertexs.
		 */
		void   Clear() { vx.clear(); vy.clear(); }

		/**	Check if a point is inside the polygon:
		 */
		bool  PointIntoPolygon(double x,double y);

		/** Computed the angle between the viewing angle and the
		 *  closest edge in the polygon to the given point.
		 */
		double ComputeTangentAngNearPoint(double x,double y,double view_phi, double min_dist);
	};

	// This must be added to any CSerializable derived class:
	DEFINE_SERIALIZABLE_POST( CPolygon )

} // End of namespace

#endif
