/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  CMEMORYSTREAM_H
#define  CMEMORYSTREAM_H

#include <MRPT/UTILS/CStream.h>
#include <MRPT/UTILS/safe_pointers.h>

/*---------------------------------------------------------------
	Class
  ---------------------------------------------------------------*/
namespace UTILS
{
	/** This CStream derived class allow using a memory buffer as a CStream.
	 *  This class is useful for storing any required set of variables or objects,
	 *   and then read them to other objects, or storing them to a file, for example.
	 *
	 * \sa CStream
	 */
	class CMemoryStream : public CStream
	{
	protected:
		 /** Method responsible for reading from the stream.
		 */
		size_t Read(void *Buffer, size_t Count);

		/** Method responsible for writing to the stream.
		 *  Write attempts to write up to Count bytes to Buffer, and returns the number of bytes actually written.
		 */
		size_t Write(const void *Buffer, size_t Count);

		/** Internal data
		 */
		void_ptr_noncopy	FMemory;
		size_t FSize, FPosition,FBytesWritten;

		/** Resizes the internal buffer size.
		 */
		void resize(size_t newSize);
	public:
		/** Default constructor
		 */
		CMemoryStream();

		/** Constructor to initilize the data in the stream from a block of memory (which is copied), and sets the current stream position at the beginning of the data.
		 */
		CMemoryStream( const void *data, const size_t nBytesInData );

		/** Destructor
		 */
		virtual ~CMemoryStream();

		/** Clears the memory buffer.
		 */
		void  Clear();

		/** Change size. This would be rarely used. Use ">>" operators for writing to stream.
		 * \sa Stream
		 */
		void  changeSize( size_t newSize );

		/** Method for moving to a specified position in the streamed resource.
		 *  \sa CStream::Seek
		 */
		size_t Seek(long Offset, CStream::TSeekOrigin Origin = sFromBeginning);

		/** Returns the total size of the internal buffer.
		 */
		size_t getTotalBytesCount();

		/** Method for getting the current cursor position, where 0 is the first byte and TotalBytesCount-1 the last one.
		 */
		size_t getPosition();

		/** Method for getting a pointer to the raw stored data.
		 * The lenght in bytes is given by getTotalBytesCount
		 */
		void*  getRawBufferData();

	}; // End of class def.

} // End of namespace

#endif
