/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CMATRIXD_H
#define CMATRIXD_H

#include <MRPT/UTILS/CSerializable.h>
#include <MRPT/UTILS/CMatrixTemplateNumeric.h>
#include <MRPT/UTILS/utils_defs.h>


namespace MRML
{
class CPose2D;
class CPose3D;
class CPoint2D;
class CPoint3D;
}

namespace UTILS
{
/**  This class is a "CSerializable" wrapper for "CMatrixTemplateNumeric<double>".
 */
class CMatrixD : public UTILS::CSerializable, public CMatrixTemplateNumeric<double>
{
	// This must be added to any CSerializable derived class:
	DEFINE_SERIALIZABLE( CMatrixD )
public:
public:
	/** Constructor
	  */
	CMatrixD(size_t row = 3, size_t col = 3) : CMatrixTemplateNumeric<double>(row,col)
	{
	}

	/** Copy constructor
	  */
	CMatrixD( const CMatrixTemplateNumeric<double> &m ) : CMatrixTemplateNumeric<double>(m)
	{
	}

	/** Copy constructor
	  */
	CMatrixD( const CMatrixFloat &m ) : CMatrixTemplateNumeric<double>(0,0)
	{
		*this = m;
	}

	/** Constructor from a MRML::CPose2D, which generates a 3x1 matrix \f$ [x y \phi]^T \f$
	   */
	CMatrixD( const MRML::CPose2D &p);

	/** Constructor from a MRML::CPose6D, which generates a 6x1 matrix \f$ [x y z yaw pitch roll]^T \f$
	   */
	CMatrixD( const MRML::CPose3D &p);

	/** Constructor from a MRML::CPoint2D, which generates a 2x1 matrix \f$ [x y]^T \f$
	   */
	CMatrixD( const MRML::CPoint2D &p);

	/** Constructor from a MRML::CPoint3D, which generates a 3x1 matrix \f$ [x y z]^T \f$
	   */
	CMatrixD( const MRML::CPoint3D &p);



	/** Assignment operator
	*/
	CMatrixD & operator = (const CMatrixTemplateNumeric<double>& m)
	{
		CMatrixTemplateNumeric<double>::operator =(m);
		return *this;
	}

	/** Assignment operator
	*/
	CMatrixD & operator = (const CMatrixFloat& m)
	{
		setSize(m.getRowCount(),m.getColCount());

		for (size_t i=0;i<m.getRowCount();i++)
			for (size_t j=0;j<m.getColCount();j++)
				(*this)(i,j) = static_cast<double>( m(i,j) );
		return *this;
	}

}; // end of class definition


	// This must be added to any CSerializable derived class:
	DEFINE_SERIALIZABLE_POST( CMatrixD )


} // End of namespace



#endif //__STD_MATRIX_H
