/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CPointPDF_H
#define CPointPDF_H

#include <MRPT/UTILS/CSerializable.h>
#include <MRPT/MRML/CPoint3D.h>

namespace MRML
{
	/** Declares a class that represents a Probability Distribution
	 *    function (PDF) of a 3D point (x,y,z).
	 *   This class is just the base class for unifying many diferent
	 *    ways this PDF can be implemented.
	 *
	 *  For convenience, a pose composition is also defined for any
	 *    PDF derived class, changeCoordinatesReference, in the form of a method rather than an operator.
     *
	 *  For a similar class for 6D poses (a 3D point with attitude), see CPose3DPDF
	 *
	 *  See also the tutorial on <a href="http://babel.isa.uma.es/mrpt/index.php/Probability_Density_Distributions_Over_Spatial_Representations">probabilistic spatial representations in the MRPT</a>.
	 *
	 * \sa CPoint3D
	 */
	class CPointPDF : public UTILS::CSerializable
	{
		DEFINE_VIRTUAL_SERIALIZABLE( CPointPDF )

	 public:
		 /** Returns an estimate of the point, (the mean, or mathematical expectation of the PDF)
		  */
		virtual CPoint3D  getEstimatedPoint() const = 0 ;

		 /** Returns an estimate of the point covariance matrix (3x3 cov.matrix for x,y,z variables)
		  */
		virtual CMatrixD  getEstimatedCovariance() const = 0 ;

		/** Copy operator, translating if necesary (for example, between particles and gaussian representations)
		  */
		virtual void  copyFrom(const CPointPDF &o) = 0;

		/** Save PDF's particles to a text file. See derived classes for more information about the format of generated files.
		 */
		virtual void  saveToTextFile(const std::string &file) const = 0;

		/** This can be used to convert a PDF from local coordinates to global, providing the point (newReferenceBase) from which
		  *   "to project" the current pdf. Result PDF substituted the currently stored one in the object.
		  */
		virtual void  changeCoordinatesReference( const CPose3D &newReferenceBase ) = 0;

		/** Draw a sample from the pdf.
		  */
		virtual void drawSingleSample(CPoint3D  &outSample) const = 0;

		/** Bayesian fusion of two point distributions (product of two distributions->new distribution), then save the result in this object (WARNING: See implementing classes to see classes that can and cannot be mixtured!)
		  * \param p1 The first distribution to fuse
		  * \param p2 The second distribution to fuse
		  * \param minMahalanobisDistToDrop If set to different of 0, the result of very separate Gaussian modes (that will result in negligible components) in SOGs will be dropped to reduce the number of modes in the output.
		  */
		virtual void  bayesianFusion( CPointPDF &p1, CPointPDF &p2, const double &minMahalanobisDistToDrop = 0)  = 0 ;

	}; // End of class def.

	DEFINE_SERIALIZABLE_POST( CPointPDF )

} // End of namespace

#endif
