/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  CDisplayWindow_H
#define  CDisplayWindow_H

#include <MRPT/UTILS/utils_defs.h>
#include <MRPT/config.h>
#include <MRPT/UTILS/SYNCH.h>

#include <MRPT/UTILS/safe_pointers.h>

/*---------------------------------------------------------------
	Class
  ---------------------------------------------------------------*/
namespace UTILS
{
	class CMRPTImage;
	class CMRPTImageFloat;


	/** This class creates a window as a graphical user interface (GUI) for displaying images to the user.
	 *
	 */
	class CDisplayWindow
	{
    public:
		/** This semaphore will be signaled when the wx window is built and ready.
		  */
		SYNCH::CSemaphore 	m_semThreadReady;

		/** This semaphore will be signaled when the wx window is destroyed.
		  */
		SYNCH::CSemaphore 	m_semWindowDestroyed;


		/** Read-only access to the wxDialog object.
		  */
		void * getWxObject() { return m_hwnd.get(); }

		/** Called by wx main thread to set m_hwnd to NULL.
		  */
		void notifyChildWindowDestruction();

	protected:
		/** The caption of the window:
		  */
		std::string		m_caption;

		/** The window's handle
		  */
		void_ptr_noncopy m_hwnd;

		/** Auxiliary
		  */
		volatile bool	m_keyPushed;

		/** Enables or disables the visualization of cursor coordinates on the window caption.
		  */
		bool			m_enableCursorCoordinates;

	public:
		/** Constructor
		 */
		CDisplayWindow( const std::string &windowCaption = std::string() );

		/** Destructor
		 */
		~CDisplayWindow();

		/** Show a given color or grayscale image on the window.
		 *  It adapts the size of the window to that of the image.
		 */
		void  showImage( const CMRPTImage	&img );

		/** Show a given grayscale image on the window.
		 *  It adapts the size of the window to that of the image.
		 */
		void  showImage( const CMRPTImageFloat	&img );

		/** Returns false if the user has already closed the window.
		  */
		bool isOpen();

		/** Resizes the window, stretching the image to fit into the display area.
		 */
		void  resize( unsigned int width, unsigned int height );

		/** Changes the position of the window on the screen.
		 */
		void  setPos( int x, int y );

		/** Waits for any key to be pushed on the image
		  */
		void  waitForKey( );

		/** Enables or disables the visualization of cursor coordinates on the window caption (default = enabled).
		  */
		void  enableCursorCoordinatesVisualization(bool enable)
		{
            m_enableCursorCoordinates = enable;
		}

		/** Changes the window title text.
		  */
		void  setWindowTitle( const std::string &str );


	}; // End of class def.

} // End of namespace

#endif
